package com.xebialabs.deployit.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.util.concurrent.atomic.AtomicReference;

import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import com.xebialabs.xlplatform.utils.SecureRandomHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.local.utils.Closeables;

public class DeployitKeyStore {

    private static final AtomicReference<KeyStore> KEYSTORE = new AtomicReference<>();

    private static final String DEPLOYIT_PASSWORD_KEY = "deployit-passsword-key";
    private static final String KEYSTORE_TYPE = "JCEKS";

    private static final String KEYSTORE_FILE = "repository-keystore";
    private static final String passwordKeyEntryPassword = "deployit";

    private DeployitKeyStore() {
    }

    public static void generateRandomKeyStore(final File parentDir, String password) {
        byte[] keyBytes = SecureRandomHolder.getRandomByteArray(32);
        SecretKey aesKey = new SecretKeySpec(keyBytes, "AES");

        // Write store to file
        createKeyStoreFile(aesKey, password, parentDir);
    }

    static void createKeyStoreFile(SecretKey key, String password, File parentDir) {
        FileOutputStream out = null;
        try {
            // Create key store and store key
            KeyStore keyStore = KeyStore.getInstance(KEYSTORE_TYPE);
            keyStore.load(null);
            keyStore.setKeyEntry(DEPLOYIT_PASSWORD_KEY, key, passwordKeyEntryPassword.toCharArray(), null);

            // Save to file
            File file = getFile(parentDir);
            logger.info("Creating key store in " + file.getAbsolutePath());
            out = new FileOutputStream(file);
                keyStore.store(out, password.toCharArray());
                KEYSTORE.set(keyStore);
        } catch (Exception e) {
            throw new DeployitKeyStoreException(e);
        } finally {
            Closeables.closeQuietly(out);
        }
    }

    public static void load(char[] password) {
        load(new File("conf"), password);
    }

    static void load(final File parentDir, final char[] password) {
        if (KEYSTORE.get() != null) {
            return;
        }
        File file = getFile(parentDir);
        if (file.exists()) {
            int tries = 0;
            char[] pwd = password;
            while (true) {
                try {
                    tries++;
                    KEYSTORE.compareAndSet(null, tryLoad(file, pwd));
                    break;
                } catch (DeployitKeyStoreException dkse) {
                    logger.debug("Could not load the encryption key for the repository.", dkse);
                    if (password.length > 0) {
                        throw dkse;
                    }

                    if (tries > 1) {
                        System.out.println("\nThe password was incorrect.");
                    }
                    if (tries <= 3) {
                        pwd = promptForPassword();
                    } else {
                        throw dkse;
                    }

                }
            }
        }
    }

    private static KeyStore tryLoad(final File file, final char[] password) {
        FileInputStream in = null;
        try {
            KeyStore keyStore = KeyStore.getInstance(KEYSTORE_TYPE);
            logger.debug("Loading password encryption key from {}", file.getAbsolutePath());
            in = new FileInputStream(file);
                keyStore.load(in, password);
                return keyStore;
        } catch (IOException | GeneralSecurityException e) {
            throw new DeployitKeyStoreException(e);
        } finally {
            Closeables.closeQuietly(in);
        }

    }

    private static char[] promptForPassword() {
        System.out.println("\nPlease enter the password to unlock the encryption key.");
        if (System.console() != null) {
            return System.console().readPassword("Password: ");
        }
        throw new DeployitKeyStoreException(
            "No Console present to read from, please ensure the password to open the XL Deploy keystore is passed on the command line");
    }

    public static SecretKey getPasswordEncryptionKey() {
        if (KEYSTORE.get() == null) {
            throw new DeployitKeyStoreException("Please load the password encryption key store before accessing a key.");
        }

        try {
            return (SecretKey) KEYSTORE.get().getKey(DEPLOYIT_PASSWORD_KEY, passwordKeyEntryPassword.toCharArray());
        } catch (Exception e) {
            throw new DeployitKeyStoreException(e);
        }
    }

    static File getFile(final File parentDir) {
        return new File(parentDir, KEYSTORE_FILE + "." + KEYSTORE_TYPE.toLowerCase());
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployitKeyStore.class);
}
