package com.xebialabs.deployit;

import com.xebialabs.deployit.util.PropertyUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;

import java.io.*;
import java.util.Map;
import java.util.Properties;

public final class ServerConfigFile {

    private final File configFile;

    public ServerConfigFile(File file) {
        configFile = file;
    }

    public boolean exists() {
        return configFile.exists();
    }

    public String getName() {
        return configFile.getName();
    }

    public void readIntoConfiguration(ServerConfiguration serverConfiguration) {
        serverConfiguration.load(readConfigurationFile(configFile));
    }

    public ServerConfiguration loadConfig() {
        return loadConfig(true, true);
    }

    public ServerConfiguration loadConfig(boolean saveDirty, boolean encryptedPasswords) {
        return loadConfig(saveDirty, encryptedPasswords, false);
    }

    public ServerConfiguration loadConfig(boolean saveDirty, boolean encryptedPasswords, boolean cloudConfigEnabled) {
        if (configFile.exists()) {
            ServerConfiguration serverConfiguration = readConfiguration(configFile, encryptedPasswords, cloudConfigEnabled);
            if (!serverConfiguration.isValid()) {
                throw new Error("Configuration " + configFile + " is not valid please run with -setup option.");
            }

            if (saveDirty) {
                saveDirtyConfiguration(serverConfiguration);
            }

            return serverConfiguration;
        } else {
            throw new Error("Configuration " + configFile
                    + " does not exist, please run with -setup option before starting the server or command line interface.");
        }
    }

    static ServerConfiguration readConfiguration(File configFile, boolean encryptedPasswords) {
        return readConfiguration(configFile, encryptedPasswords, false);
    }

    static ServerConfiguration readConfiguration(File configFile, boolean encryptedPasswords, boolean cloudConfigEnabled) {
        ServerConfiguration serverConfiguration = new ServerConfiguration();
        serverConfiguration.setSpringCloudEnabled(cloudConfigEnabled);
        serverConfiguration.setFile(configFile);
        serverConfiguration.setConfigResource(new ClassPathResource(configFile.getName()));
        serverConfiguration.load(readConfigurationFile(configFile), encryptedPasswords);
        return serverConfiguration;
    }

    static Properties readConfigurationFile(File configFile) {
        Properties properties = new Properties();
        try (InputStream inputStream = new FileInputStream(configFile)) {
            properties.load(inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        readSystemEnvironmentVariables(properties);
        return properties;
    }

    static Properties readSystemEnvironmentVariables(Properties properties) {
        System.getenv().entrySet().stream()
                .filter(m -> m.getKey().startsWith("XL_SPRING_CLOUD_"))
                .forEach(p -> properties.setProperty(
                        p.getKey().replaceAll("_", ".").toLowerCase(), p.getValue()));
        return properties;
    }

    public void appendConfiguration(Map<String, String> additionalProperties) {
        Properties properties = new Properties();
        try (InputStream input = new FileInputStream(configFile);
             OutputStream output = new BufferedOutputStream(new FileOutputStream(configFile, true))) {
            Properties existingProperties = new Properties();
            existingProperties.load(input);

            additionalProperties
                    .entrySet()
                    .stream()
                    .filter((e) -> !existingProperties.containsKey(e.getKey()))
                    .forEach((e) -> properties.setProperty(e.getKey(), e.getValue()));

            if (!properties.isEmpty()) {
                properties.store(output, null);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void writeConfiguration(ServerConfiguration serverConfiguration) {
        Properties properties = new Properties();
        serverConfiguration.save(properties);
        PropertyUtil.storePropertiesFile(configFile, properties);
    }

    public void saveDirtyConfiguration(ServerConfiguration serverConfiguration) {
        if (serverConfiguration.isDirty()) {
            logger.info("Saving (dirty) configuration");
            writeConfiguration(serverConfiguration);
        }
    }

    @Override
    public String toString() {
        return configFile.getAbsolutePath();
    }

    private static final Logger logger = LoggerFactory.getLogger(ServerConfigFile.class);
}
