package com.xebialabs.deployit.core.xml;

import java.util.*;
import java.util.stream.Collectors;

import com.xebialabs.deployit.core.*;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.StringValueConverter;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;

import static com.xebialabs.deployit.repository.StringValueConverter.valueToString;
import static java.util.stream.Collectors.toList;

public class PasswordEncryptingCiConverter extends ConfigurationItemConverter {

    private final PasswordEncrypter passwordEncrypter;
    private final StringValueConverter converter;

    public PasswordEncryptingCiConverter() {
        this.passwordEncrypter = PasswordEncrypter.getInstance();
        this.converter = new StringValueConverter(passwordEncrypter);

        setWriteValidationMessages(true);
    }

    //
    // Write
    //

    @Override
    protected void writeStringProperty(Object value, PropertyDescriptor propertyDescriptor, CiWriter writer) {
        if (propertyDescriptor.isPassword()) {
            writer.valueAsString(passwordEncrypter.ensureEncrypted(value.toString()));
        } else {
            super.writeStringProperty(value, propertyDescriptor, writer);
        }
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    @Override
    protected void writeCollectionOfStringProperty(Object value, PropertyDescriptor propertyDescriptor, CiWriter writer) {
        if (value instanceof AbstractStringView) {
            if (propertyDescriptor.isPassword()) {
                value = ((AbstractStringView) value).encrypt();
            }
            Collection<StringValue> wrapped = ((AbstractStringView) value).getWrapped();
            value = wrapped.stream().map(v -> valueToString(passwordEncrypter, true).apply(v)).collect(toList());
        }
        super.writeCollectionOfStringProperty(value, propertyDescriptor, writer);
    }

    @Override
    protected void writeMapStringStringProperty(Object value, PropertyDescriptor propertyDescriptor, CiWriter writer) {
        if (value instanceof MapStringStringView) {
            if (propertyDescriptor.isPassword()) {
                value = ((MapStringStringView) value).encrypt();
            }
            Map<String, StringValue> wrapped = ((MapStringStringView) value).getWrapped();
            value = wrapped.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> valueToString(passwordEncrypter, true).apply(e.getValue())));
        }
        super.writeMapStringStringProperty(value, propertyDescriptor, writer);
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void writeCollectionOfCiProperty(Object value, PropertyDescriptor propertyDescriptor, CiWriter writer) {
        // To the remote side, we only write ids
        Collection<ConfigurationItem> cis = (Collection<ConfigurationItem>) value;
        Collection<String> ids = cis.stream().map(ConfigurationItem::getId).filter(Objects::nonNull).collect(toList());
        writer.ciReferences(ids);
    }

    //
    // Read
    //

    @Override
    protected void readStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        if (propertyDescriptor.isPassword()) {
            propertyDescriptor.set(configurationItem, passwordEncrypter.ensureDecrypted(reader.getStringValue()));
        } else {
            super.readStringProperty(configurationItem, propertyDescriptor, reader);
        }
    }

    @Override
    protected void readCollectionOfStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        AbstractStringView<?> strings;

        if (propertyDescriptor.getKind() == PropertyKind.SET_OF_STRING) {
            strings = new SetOfStringView(readConverted(reader, new LinkedHashSet<>()));
        } else {
            strings = new ListOfStringView(readConverted(reader, new ArrayList<>()));
        }

        // Ensure all are encrypted when the propertyDescriptor is password.
        if (propertyDescriptor.isPassword()) {
            strings = strings.encrypt();
        }

        propertyDescriptor.set(configurationItem, strings);

    }

    private <T extends Collection<StringValue>> T readConverted(CiReader reader, T convertedStrings) {
        List<String> rawStrings = reader.getStringValues();
        for (String string : rawStrings) {
            convertedStrings.add(converter.convert(string));
        }

        return convertedStrings;
    }

    @Override
    protected void readMapStringStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        Map<String, StringValue> map = new HashMap<>();
        @SuppressWarnings("deprecation")
        boolean encrypt = configurationItem.getType().equals(Type.valueOf(com.xebialabs.deployit.plugin.api.udm.EncryptedDictionary.class));
        for (Map.Entry<String, String> entry : reader.getStringMap().entrySet()) {
            map.put(entry.getKey(), converter.convert(entry.getValue(), encrypt));
        }

        MapStringStringView value = new MapStringStringView(map);
        if (propertyDescriptor.isPassword()) {
            value = value.encrypt();
        }

        propertyDescriptor.set(configurationItem, value);
    }
}
