package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.core.rest.api.support.PaginationSupport;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.api.security.User;
import com.xebialabs.deployit.engine.spi.event.UserCreatedEvent;
import com.xebialabs.deployit.engine.spi.event.UserDeletedEvent;
import com.xebialabs.deployit.engine.spi.event.UserPasswordChangedEvent;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.UserService;
import org.jboss.resteasy.spi.HttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.core.Context;
import java.util.List;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;

@Controller
public class UserResource extends AbstractSecuredResource implements com.xebialabs.deployit.engine.api.UserService {

    @Context
    private HttpResponse response;

    @Autowired
    UserService userService;

    @Override
    public User create(final String username, final User user) {
        checkPermission(EDIT_SECURITY);
        sanityCheckUser(username, user);

        logger.debug("Creating user {}", username);
        userService.create(username, user.getPassword());
        com.xebialabs.deployit.security.User u = userService.read(username);
        logger.debug("Created user {}", u.getUsername());

        EventBusHolder.publish(new UserCreatedEvent(username));

        return new User(u.getUsername(), u.isAdmin());
    }

    @Override
    public User read(final String username) {
        checkPermission(EDIT_SECURITY);

        com.xebialabs.deployit.security.User u = userService.read(username);

        return new User(u.getUsername(), u.isAdmin());
    }

    @Override
    public List<String> listUserNames(final String username, final Paging paging, final Ordering order) {
        checkPermission(EDIT_SECURITY);
        List<String> usernames = userService.listUsernames(username, paging, order);
        PaginationSupport.addTotalCountHeader(userService.countUsers(username), response);
        return usernames;
    }

    @Override
    public void modifyPassword(final String username, final User user) {
        checkPermission(EDIT_SECURITY);
        sanityCheckUser(username, user);

        userService.modifyPassword(username, user.getPassword());

        EventBusHolder.publish(new UserPasswordChangedEvent(username));
    }

    @Override
    public void delete(final String username) {
        checkPermission(EDIT_SECURITY);

        userService.delete(username);

        EventBusHolder.publish(new UserDeletedEvent(username));
    }

    private static void sanityCheckUser(final String username, final User user) {
        if (user.getUsername() == null || user.getUsername().trim().isEmpty()) {
            throw new IllegalArgumentException("User name cannot be empty");
        }
        if (!user.getUsername().equals(username)) {
            throw new IllegalArgumentException("Username in URL is " + username + " but username in POST data is " + user.getUsername());
        }
    }

    void setUserService(UserService userService) {
        this.userService = userService;
    }

    private static final Logger logger = LoggerFactory.getLogger(UserResource.class);
}
