package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.core.rest.api.support.PaginationSupport;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.PermissionService;
import com.xebialabs.deployit.engine.api.RepositoryService;
import com.xebialabs.deployit.engine.api.dto.CIPermissionDto;
import com.xebialabs.deployit.engine.api.dto.EditRolePermissionsDto;
import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.spi.event.PermissionGrantedEvent;
import com.xebialabs.deployit.engine.spi.event.PermissionRevokedEvent;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.*;
import com.xebialabs.deployit.security.permission.Permission;
import jakarta.ws.rs.core.Context;
import org.jboss.resteasy.spi.HttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Controller;

import java.util.*;
import java.util.stream.Collectors;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.VIEW_SECURITY;

@Controller
public class PermissionResource extends AbstractSecuredResource implements PermissionService {

    @Autowired
    private RoleService roleService;

    @Autowired
    private PermissionEditor permissionEditor;

    @Autowired
    private PermissionLister permissionLister;

    @Autowired
    private RepositoryService repositoryService;
    private DtoWriter dtoWriter = new DtoWriter();

    @Context
    private HttpResponse response;



    private static final Logger logger = LoggerFactory.getLogger(PermissionResource.class);

    private static Permission getPermission(final String permissionName) {
        final Permission permission = Permission.find(permissionName);
        if (permission == null) {
            throw new UnknownPermissionException(permissionName);
        }
        return permission;
    }

    private HashMap<String, Collection<String>> getStringCollectionHashMap(final Map<String, List<String>> stringStringMultimap) {
        return new HashMap<>(stringStringMultimap.entrySet().stream()
                .filter(e -> !e.getValue().isEmpty())
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    }

    @Override
    public Map<String, Collection<String>> getGrantedPermissions(String roleName) {
        Role role = roleService.getRoleForRoleName(roleName);
        return getStringCollectionHashMap(permissionLister.listPermissions(role));
    }

    @Override
    public Map<String, Collection<String>> getGrantedPermissionsById(String roleId) {
        Role role = roleService.getRoleForRoleId(roleId);
        return getStringCollectionHashMap(permissionLister.listPermissions(role));
    }

    @Override
    public Map<String, Collection<String>> getMyGrantedPermissions() {
        Authentication authentication = Permissions.getAuthentication();
        List<Role> roles = roleService.getRolesFor(authentication);
        return getStringCollectionHashMap(permissionLister.listPermissions(roles));
    }

    @Override
    public boolean isGranted(String permissionName, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        Permission permission = Permission.find(permissionName);
        Map<Role, Set<Permission>> rolePermissionMultimap = permissionEditor.readPermissions(id);
        return rolePermissionMultimap.computeIfAbsent(role, r -> new HashSet<>()).contains(permission);
    }

    @Override
    public void grant(String permission, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        permissionEditor.grant(role, getPermission(permission), id);
        EventBusHolder.publish(new PermissionGrantedEvent(id, roleName, permission));
    }

    @Override
    public void revoke(String permission, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        permissionEditor.revoke(role, getPermission(permission), id);
        EventBusHolder.publish(new PermissionRevokedEvent(id, roleName, permission));
    }

    @Override
    public boolean isGrantedToMe(String permission, String id) {
        return getPermission(permission).getPermissionHandler().hasPermission(id);
    }

    @Override
    public void updatePermissions(String onConfigurationItem, List<EditRolePermissionsDto> rolesWithPermissions) {
        checkPermission(EDIT_SECURITY);
        Map<Role, List<Permission>> addedPermissions = new HashMap<>();
        Map<Role, List<Permission>> removedPermissions = new HashMap<>();
        rolesWithPermissions.forEach(r -> {
            addedPermissions.put(roleService.getRoleForRoleName(r.getRoleName()), r.getAddedPermissions().stream().map(p -> getPermission(p)).collect(Collectors.toList()));
            removedPermissions.put(roleService.getRoleForRoleName(r.getRoleName()), r.getRemovedPermissions().stream().map(p -> getPermission(p)).collect(Collectors.toList()));
        });
        permissionEditor.updatePermissions(onConfigurationItem, addedPermissions, removedPermissions);
    }

    @Override
    public void inheritParentPermissions(String id, CIPermissionDto request) {
        checkPermission(EDIT_SECURITY);
        boolean inheritParentPermissions = request.isInheritParentPermission();
        boolean isSecureCi = repositoryService.isSecure(id);
        List<EditRolePermissionsDto> editRolePermissionsDtoList = request.getEditRolePermissionsDtoList();
        if (!inheritParentPermissions) {
            if (editRolePermissionsDtoList == null || editRolePermissionsDtoList.isEmpty()) {
                throw new EmptyPermissionException();
            }
            this.updatePermissions(id, editRolePermissionsDtoList);
        } else if (shouldUpdate(true, isSecureCi)) {
            permissionEditor.updatePermissions(id, new HashMap<>(), new HashMap<>());
        } else{
            logger.debug("Skipping update of permissions for CI {} as inheritParentPermissions={} and isSecureCi={}", id, inheritParentPermissions, isSecureCi);
        }
    }

    @Override
    public CIPermissionDto getInheritParentPermissionsById(String id) {
        checkPermission(EDIT_SECURITY);
        boolean isSecureCi = repositoryService.isSecure(id);
        CIPermissionDto dto = new CIPermissionDto();
        dto.setInheritParentPermission(!isSecureCi);
        dto.setEditRolePermissionsDtoList(null);
        return dto;
    }

    @Override
    public Map<String, Object> getIsInheritedParentAndPermissions(String id, String rolePattern, Paging paging, Ordering order, Boolean includeInherited) {
        checkPermissions(EDIT_SECURITY);
        checkPermissions(VIEW_SECURITY);
        boolean isSecureCi = repositoryService.isSecure(id);
        String rootCI = id.split("/")[0];
        Map<Role, Set<Permission>> rolePermissionMultimap;
        if (rolePattern == null && (paging == null || paging.resultsPerPage() == -1) && order == null) {
            rolePermissionMultimap = permissionEditor.readPermissions(id, includeInherited);
        } else {
            rolePermissionMultimap = permissionEditor.readPermissions(id, rolePattern, paging, order, includeInherited);
            PaginationSupport.addTotalCountHeader(roleService.countRoles(id, rolePattern), response);
        }

        return dtoWriter.writeAllPermissionsForRole(rolePermissionMultimap, isSecureCi, rootCI);
    }


    private boolean shouldUpdate(boolean inheritParentPermissions, boolean isSecureCi) {
        return inheritParentPermissions == isSecureCi;
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class UnknownPermissionException extends DeployitException {
        public UnknownPermissionException(String permission) {
            super("Permission %s does not exist.", permission);
        }
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class EmptyPermissionException extends DeployitException {
        public EmptyPermissionException() {
            super("RolePermissions must be defined if inheritParentPermissions is false");
        }
    }
}
