package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.PermissionService;
import com.xebialabs.deployit.engine.spi.event.PermissionGrantedEvent;
import com.xebialabs.deployit.engine.spi.event.PermissionRevokedEvent;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.*;
import com.xebialabs.deployit.security.permission.Permission;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Controller;

import java.util.*;
import java.util.stream.Collectors;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;

@Controller
public class PermissionResource extends AbstractSecuredResource implements PermissionService {

    @Autowired
    private RoleService roleService;

    @Autowired
    private PermissionEditor permissionEditor;

    @Autowired
    private PermissionLister permissionLister;

    private static Permission getPermission(final String permissionName) {
        final Permission permission = Permission.find(permissionName);
        if (permission == null) {
            throw new UnknownPermissionException(permissionName);
        }
        return permission;
    }

    private HashMap<String, Collection<String>> getStringCollectionHashMap(final Map<String, List<String>> stringStringMultimap) {
        return new HashMap<>(stringStringMultimap.entrySet().stream()
                .filter(e -> !e.getValue().isEmpty())
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    }

    @Override
    public Map<String, Collection<String>> getGrantedPermissions(String roleName) {
        Role role = roleService.getRoleForRoleName(roleName);
        return getStringCollectionHashMap(permissionLister.listPermissions(role));
    }

    @Override
    public Map<String, Collection<String>> getGrantedPermissionsById(String roleId) {
        Role role = roleService.getRoleForRoleId(roleId);
        return getStringCollectionHashMap(permissionLister.listPermissions(role));
    }

    @Override
    public Map<String, Collection<String>> getMyGrantedPermissions() {
        Authentication authentication = Permissions.getAuthentication();
        List<Role> roles = roleService.getRolesFor(authentication);
        return getStringCollectionHashMap(permissionLister.listPermissions(roles));
    }

    @Override
    public boolean isGranted(String permissionName, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        Permission permission = Permission.find(permissionName);
        Map<Role, Set<Permission>> rolePermissionMultimap = permissionEditor.readPermissions(id);
        return rolePermissionMultimap.computeIfAbsent(role, r -> new HashSet<>()).contains(permission);
    }

    @Override
    public void grant(String permission, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        permissionEditor.grant(role, getPermission(permission), id);
        EventBusHolder.publish(new PermissionGrantedEvent(id, roleName, permission));
    }

    @Override
    public void revoke(String permission, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        permissionEditor.revoke(role, getPermission(permission), id);
        EventBusHolder.publish(new PermissionRevokedEvent(id, roleName, permission));
    }

    @Override
    public boolean isGrantedToMe(String permission, String id) {
        return getPermission(permission).getPermissionHandler().hasPermission(id);
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class UnknownPermissionException extends DeployitException {
        public UnknownPermissionException(String permission) {
            super("Permission %s does not exist.", permission);
        }
    }
}
