package com.xebialabs.deployit.core.rest.api;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.api.InternalHistoryProxy;
import com.xebialabs.deployit.core.api.dto.Comparison;
import com.xebialabs.deployit.core.api.dto.Revision;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.ConfigurationItemRevision;
import com.xebialabs.deployit.repository.HistoryService;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.comparison.Comparator;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

@Controller
public class HistoryResource implements InternalHistoryProxy {

    @Autowired
    private HistoryService historyService;

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private Comparator comparator;

    @Override
    public List<Revision> readRevisions(final String id) {
        checkId(id);

        List<ConfigurationItemRevision> versionRevisions = historyService.getVersionRevisions(id);
        return versionRevisions.stream().map(r -> new Revision(r.getRevisionId(), r.getCreatedAt(), r.getUsername())).collect(Collectors.toList());
    }

    @Override
    public ConfigurationItem readRevision(final String id, final String revisionName) {
        checkId(id);
        return _readRevision(id, revisionName);
    }

    @Override
    public Comparison compare(final String id, final List<String> revisionNames) {
        checkId(id);
        checkArgument(revisionNames.size() > 1, "Need two revisions to compare.");

        String firstRevision = revisionNames.get(0);
        ConfigurationItem configurationItem = _readRevision(id, firstRevision);

        final Map<String, List<String>> multimap = comparator.compare(configurationItem, revisionNames.stream().skip(1).map(r -> _readRevision(id, r)).collect(Collectors.toList()));

        return Comparison.from(multimap);
    }

    private ConfigurationItem _readRevision(final String id, final String revisionName) {
        if (revisionName.equals(HistoryService.CURRENT_REVISION)) {
            return repositoryService.read(id);
        }
        return historyService.readRevision(id, revisionName);
    }

    private void checkId(final String id) {
        checkNotNull(id, "id");
        if (!repositoryService.exists(id)) {
            throw new Checks.IncorrectArgumentException("The id [%s] does not exist.", id);
        }
    }
}
