package com.xebialabs.deployit.core.rest.api;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Paths;
import java.util.*;

import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.authentication.viewas.ViewAsData;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.booter.local.PluginVersions;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.ServerService;
import com.xebialabs.deployit.engine.api.dto.ServerInfo;
import com.xebialabs.deployit.engine.api.dto.ServerPluginInfo;
import com.xebialabs.deployit.engine.api.dto.ServerState;
import com.xebialabs.deployit.engine.spi.event.*;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.event.ShutdownEvent;
import com.xebialabs.deployit.security.authentication.viewas.ViewAsAuthenticationFinder;
import com.xebialabs.deployit.service.gc.GarbageCollectionService;
import com.xebialabs.license.service.LicenseService;

import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static java.util.stream.Collectors.toList;

@Controller
public class ServerResource extends AbstractSecuredResource implements ServerService {

    @Autowired
    private GarbageCollectionService garbageCollectionService;

    @Autowired
    private LicenseService licenseService;

    @Autowired
    private List<ViewAsAuthenticationFinder> viewAsAuthenticationFinders;

    @Autowired
    private RoleService roleService;

    @Autowired
    ViewAsData viewAsData;

    private final Map<String, String> productLogos = new HashMap<>() {{
        put("xl-deploy", "xl-deploy-logo.svg");
        put("xl-release", "Digital-ai-Release.svg");
    }};

    @Override
    public void shutdown() {
        checkPermission(ADMIN);
        EventBusHolder.publish(new SystemStoppedEvent());
        EventBusHolder.publish(new ShutdownEvent());
    }

    @Override
    public void startMaintenance() {
        checkPermission(ADMIN);
        EventBusHolder.publish(new MaintenanceStartEvent());
    }

    @Override
    public void stopMaintenance() {
        checkPermission(ADMIN);
        EventBusHolder.publish(new MaintenanceStopEvent());
    }

    @Override
    public ServerState getState() {
        // NORMAL, MAINTENANCE
        String currentMode = com.xebialabs.deployit.ServerState.getInstance().getCurrentMode().toString();
        ServerState state = new ServerState();
        state.setCurrentMode(currentMode);
        return state;
    }


    @Override
    public ServerInfo getInfo() {
        ClassPathResource resource = new ClassPathResource("com/xebialabs/deployit/release.properties");
        Properties release = new Properties();
        try {
            release.load(resource.getInputStream());
            ServerInfo si = new ServerInfo();
            si.setVersion(release.getProperty("version"));
            for (String plugin : PluginVersions.getRegisteredPlugins()) {
                si.getPluginsInfo().add(new ServerPluginInfo(plugin, PluginVersions.getVersionFor(plugin)));
            }
            si.setClasspathEntries(getServerClassPath());
            return si;
        } catch (IOException e) {
            throw new DeployitException(e);
        }
    }

    @Override
    public void logout() {
        // Is taken care of by BasicAuthWithRememberMeFilter
    }

    @Override
    public void viewAs(final String username, final List<String> roles) {
        checkPermission(ADMIN);
        if (username != null && roles != null && !roles.isEmpty()) {
            throw new RuntimeException("Use the View As feature with either username or roles");
        }
        viewAsData.setUser(null);
        viewAsData.setRoles(null);
        if (username != null && !username.equals("")) {
            viewAsData.setUser(authenticateViewAs(username));
        } else if (roles != null && !roles.isEmpty()) {
            viewAsData.setRoles(roleService.getRoles().stream().filter(r -> roles.contains(r.getName())).collect(toList()));
        }
    }

    private Authentication authenticateViewAs(final String username) {
        for (ViewAsAuthenticationFinder viewAsAuthentication : viewAsAuthenticationFinders) {
            Authentication authentication = viewAsAuthentication.search(username);
            if (authentication != null) {
                return authentication;
            }
        }
        throw new RuntimeException("Unknown user: '" + username + "'.");
    }

    private static List<String> getServerClassPath() {
        String classpath = System.getProperty("java.class.path");
        return Arrays.asList(classpath.split(System.getProperty("path.separator")));
    }

    @Override
    public void gc() {
        checkPermission(ADMIN);
        garbageCollectionService.runGarbageCollector();
        EventBusHolder.publish(new RepositoryGarbageCollectedEvent());
    }

    @Override
    public void licenseReload() {
        checkPermission(ADMIN);
        licenseService.reload();
        EventBusHolder.publish(new LicenseReloadCollectedEvent());
    }

    @Override
    public Response getLogo(String productName) {
        String logoFileName = productLogos.get(productName);
        if (logoFileName == null) {
            return Response.status(Response.Status.NOT_FOUND).build();
        }

        InputStream is;
        try {
            File logo = Paths.get("conf", logoFileName).toFile();

            if (logo.exists()) {
                is = new FileInputStream(logo);
            } else {
                is = this.getClass().getClassLoader().getResourceAsStream("web/icons/" + logoFileName);
            }

            if (is != null) {
                StreamingOutput output = stream -> {
                    stream.write(IOUtils.toByteArray(is));
                    is.close();
                };
                return Response.ok(output, "image/svg+xml").build();
            }
        } catch (IOException ignored) {

        }
        return Response.status(Response.Status.NOT_FOUND).build();
    }
}
