package com.xebialabs.xlplatform.settings

import akka.actor.{ExtendedActorSystem, Extension, ExtensionId, ExtensionIdProvider}
import akka.util.Timeout
import com.typesafe.config.Config

import java.io.File
import scala.concurrent.duration._
import scala.language.postfixOps

object CommonSettings extends ExtensionId[CommonSettings] with ExtensionIdProvider {

  override def createExtension(system: ExtendedActorSystem): CommonSettings = new CommonSettings(system.settings.config)

  override def lookup(): ExtensionId[_ <: Extension] = CommonSettings
}

class CommonSettings(config: Config) extends Extension with shared.CommonSettings {
  lazy val tasker = new TaskerSettings(config)
  lazy val satellite = new SatelliteSettings(config.getConfig("xl"))
  lazy val server = new ServerSettings(config.getConfig("xl"))
  lazy val worker = new WorkerSettings(config.getConfig("xl.worker"))
  lazy val scheduler = new SchedulerSettings(config) // get rid
  lazy val security = new SecuritySettings(config.getConfig("xl.server.ssl"))
  lazy val inProcessTaskEngine: Boolean = config.getBoolean("deploy.task.in-process-worker")
  lazy val algorithm = new AlgorithmSettings(config.getConfig("xl.algorithms"))
  lazy val session = new SessionSettings(config.getConfig("xl.server.session"))
}

class TaskerSettings(val config: Config) extends SettingsSupport with shared.TaskerSettings {
  import SettingsSupport._
  lazy val recoveryDir = new File(value[String]("deploy.task.recovery-dir"))
  lazy val maxActiveTasks: Int = value[Int]("deploy.task.max-active-tasks")
  lazy val shutdownTimeout: FiniteDuration = value[FiniteDuration]("deploy.task.shutdown-timeout")
  lazy val tickDuration: FiniteDuration = value[FiniteDuration]("akka.scheduler.tick-duration")
  lazy val stepRetryDelay: FiniteDuration =  value[FiniteDuration]("deploy.task.step.retry-delay")
  lazy val enableCopyArtifactRetry: Boolean = value[Boolean]("deploy.task.step.on-copy-artifact.enable-retry")
  lazy val askTimeout: Timeout = valueWithDefault("xl.tasker.askTimeout", 10 seconds)
  lazy val stepRunTimeout: FiniteDuration = valueWithDefault("deploy.task.step.run-timeout", 6 hours)
  lazy val stepRetryCount: Int = valueWithDefault("deploy.task.step.max-retry-number", 4320) // 6 hours of default timeout / retry every 5 second
  lazy val asyncDnsEnabled: Boolean = valueOption[String]("akka.io.dns.resolver").exists(r => r.equals("async-dns"))
}

class SatelliteSettings(val config: Config) extends SettingsSupport with shared.SatelliteSettings {
  import SettingsSupport._
  lazy val uploadIdleTimeout: Timeout = valueWithDefault("satellite.timeout.upload.idle", 10 seconds)
  lazy val infoTimeout: Timeout = valueWithDefault("satellite.timeout.info", 10 seconds)
  lazy val pingTimeout: Timeout = valueWithDefault("satellite.timeout.ping", 5 seconds)
  lazy val remoteAskTimeout: Timeout = valueWithDefault("satellite.timeout.remoteAsk", 60 seconds)
  lazy val enabled: Boolean = valueWithDefault("satellite.enabled", false)
  lazy val streamingConnectionTimeout: Timeout = valueWithDefault("satellite.timeout.streaming", 10 seconds)
  lazy val maxConcurrentUploads: Int = valueWithDefault("satellite.streaming.max-uploads", 10)
  lazy val fileUploadAttempts: Int = valueWithDefault("satellite.streaming.upload-attempts", 3)
}

class ServerSettings(val config: Config) extends SettingsSupport with shared.ServerSettings {
  import SettingsSupport._
  lazy val hostname: String = value[String]("server.hostname")
  lazy val aggregationTimeout: FiniteDuration = valueWithDefault("server.aggregation-timeout", 9 seconds)
}

class WorkerSettings(val config: Config) extends SettingsSupport with  shared.WorkerSettings {
  import SettingsSupport._
  lazy val connectInterval: FiniteDuration = valueWithDefault("connect.interval", 10 seconds)
  lazy val reconnectDelay: FiniteDuration = valueWithDefault("reconnect.delay", 30 seconds)
  lazy val reconnectInterval: FiniteDuration = valueWithDefault("reconnect.interval", 10 seconds)
}

class AlgorithmSettings(val config: Config) extends SettingsSupport with shared.AlgorithmSettings {
  import SettingsSupport._
  lazy val checksum: String = valueWithDefault("checksum", "SHA-256")
}

// get rid
class SchedulerSettings(val config: Config) extends SettingsSupport with shared.SchedulerSettings {
  import SettingsSupport._
  lazy val tickDuration: FiniteDuration = value[FiniteDuration]("akka.scheduler.tick-duration")
}

class SessionSettings(val config: Config) extends SettingsSupport with shared.SessionSettings {
  import SettingsSupport._
  lazy val cleanupCron: String = value[String]("cleanup-cron")
}
