package com.gradle.develocity.agent.gradle.scan;

import com.gradle.obfuscation.Keep;
import org.gradle.api.Action;
import org.gradle.api.provider.Property;

/**
 * The configuration for Build Scan publication.
 *
 * @since 3.17
 */
@Keep
public interface BuildScanConfiguration {

    /**
     * Executes the given action in a background thread, allowing the current Gradle work to continue.
     * <p>
     * This method is useful for capturing values, tags and links that are expensive to compute.
     * By capturing them in the background, Gradle can continue doing other work, making your build faster.
     * <p>
     * For example, if you are capturing the Git commit ID as a custom value you should invoke Git in the background:
     * <pre>
     * develocity.buildScan.background {
     *   def commitId = 'git rev-parse --verify HEAD'.execute().text.trim()
     *   value "Git Commit ID", commitId
     * }
     * </pre>
     * <p>
     * All background work will be completed before finishing the build and publishing the build scan.
     * <p>
     * Any errors that are thrown by the background action will be logged and captured in the build scan.
     *
     * @param action the action to execute in the background
     */
    void background(Action<? super BuildScanConfiguration> action);

    /**
     * Captures a tag for the current build. The tag is not required to be unique for a given build.
     *
     * @param tag the name of the tag, must not be null
     */
    void tag(String tag);

    /**
     * Captures a named value for the current build. The name is not required to be unique for a given build.
     *
     * @param name  the name, must not be null
     * @param value the value, may be null
     */
    void value(String name, String value);

    /**
     * Captures a named link for the current build. The name is not required to be unique for a given build.
     *
     * @param name the name, must not be null
     * @param url  the url, must not be null
     */
    void link(String name, String url);

    /**
     * Registers a callback that is invoked when the build has finished, but <i>before</i> this extension stops
     * accepting to be called.
     *
     * @param action the action to execute when the build has finished
     */
    void buildFinished(Action<? super BuildResult> action);

    /**
     * Registers a callback that is invoked when a build scan has been published successfully.
     * <p>
     * If {@link #getUploadInBackground()} is {@code true}, the callback is executed before the build scan is uploaded.
     * If the build scan is accessed immediately, it may not be available yet as it is still being uploaded to the server.
     *
     * @param action the action to execute when the build scan has been published successfully
     */
    void buildScanPublished(Action<? super PublishedBuildScan> action);

    /**
     * The location of the Gradle Terms of Use that are agreed to when creating a build scan.
     */
    Property<String> getTermsOfUseUrl();

    /**
     * The agreement of the Gradle Terms of Use.
     */
    Property<String> getTermsOfUseAgree();

    /**
     * Specifies whether to upload the build scan in background after the build has finished.
     * <p>
     * Defaults to {@code true}.
     * <p>
     * This allows the build to finish sooner, but can be problematic in build environments that
     * terminate as soon as the build is finished as the upload may be terminated before it completes.
     * Background uploading should be disabled for such environments.
     * <p>
     * This setting can also be set by the {@code scan.uploadInBackground} system property,
     * which, if set, takes precedence over any value set by this method and the default.
     */
    Property<Boolean> getUploadInBackground();

    /**
     * Allows configuring when a build scan should be published at the end of the build.
     *
     * @return the publishing configuration
     */
    BuildScanPublishingConfiguration getPublishing();

    /**
     * Allows configuring when a build scan should be published at the end of the build.
     */
    void publishing(Action<? super BuildScanPublishingConfiguration> action);

    /**
     * Allows registering functions for obfuscating certain identifying information within build scans.
     *
     * @return the register of obfuscation functions
     */
    BuildScanDataObfuscationConfiguration getObfuscation();

    /**
     * Allows registering functions for obfuscating certain identifying information within build scans.
     *
     * @param action a function to be applied to the register of obfuscation functions
     */
    void obfuscation(Action<? super BuildScanDataObfuscationConfiguration> action);

    /**
     * Allows configuring what data will be captured as part of the build scan.
     *
     * @return the capture configuration
     */
    BuildScanCaptureConfiguration getCapture();

    /**
     * Allows configuring what data will be captured as part of the build scan.
     *
     * @param action a function to be applied to the capture settings
     */
    void capture(Action<? super BuildScanCaptureConfiguration> action);

}
