/*
 * Decompiled with CFR 0.152.
 */
package com.bmuschko.gradle.docker.internal;

import com.bmuschko.gradle.docker.DockerRegistryCredentials;
import com.bmuschko.gradle.docker.shaded.com.fasterxml.jackson.core.JsonProcessingException;
import com.bmuschko.gradle.docker.shaded.com.fasterxml.jackson.databind.ObjectMapper;
import com.bmuschko.gradle.docker.shaded.com.fasterxml.jackson.databind.json.JsonMapper;
import com.github.dockerjava.api.model.AuthConfig;
import com.github.dockerjava.api.model.AuthConfigurations;
import com.github.dockerjava.core.NameParser;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import javax.inject.Inject;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.process.ExecOperations;

public class RegistryAuthLocator {
    private static final String DOCKER_CONFIG = "DOCKER_CONFIG";
    private static final String USER_HOME = "user.home";
    private static final String DOCKER_DIR = ".docker";
    private static final String CONFIG_JSON = "config.json";
    private static final String AUTH_SECTION = "auths";
    private static final String HELPERS_SECTION = "credHelpers";
    private static final String CREDS_STORE_SECTION = "credsStore";
    private static final String DEFAULT_HELPER_PREFIX = "docker-credential-";
    private Logger logger = Logging.getLogger(RegistryAuthLocator.class);
    private final ObjectMapper objectMapper = JsonMapper.builder().build();
    private final File configFile;
    private final String commandPathPrefix;
    private final String helperSuffix;
    private final ExecOperations execOperations;

    private RegistryAuthLocator(ExecOperations execOperations, File configFile, String commandPathPrefix, String helperSuffix) {
        this.execOperations = execOperations;
        this.configFile = configFile;
        this.commandPathPrefix = commandPathPrefix;
        this.helperSuffix = helperSuffix;
    }

    private RegistryAuthLocator(ExecOperations execOperations, File configFile) {
        this(execOperations, configFile, DEFAULT_HELPER_PREFIX, "");
    }

    private RegistryAuthLocator(ExecOperations execOperations) {
        this(execOperations, new File(RegistryAuthLocator.configLocation()), DEFAULT_HELPER_PREFIX, "");
    }

    AuthConfig lookupAuthConfigWithDefaultAuthConfig(String image) {
        return this.lookupAuthConfigWithAuthConfig(image, new AuthConfig());
    }

    public AuthConfig lookupAuthConfig(String image, DockerRegistryCredentials registryCredentials) {
        AuthConfig defaultConfig = this.createAuthConfig(registryCredentials);
        if (this.isProvidedByBuild(defaultConfig)) {
            return defaultConfig;
        }
        return this.lookupAuthConfigWithAuthConfig(image, defaultConfig);
    }

    private boolean isProvidedByBuild(AuthConfig defaultConfig) {
        return defaultConfig.getRegistryAddress() != null && defaultConfig.getUsername() != null && defaultConfig.getPassword() != null;
    }

    private AuthConfig lookupAuthConfigWithAuthConfig(String image, AuthConfig defaultAuthConfig) {
        AuthConfig authConfigForRegistry = this.lookupAuthConfigForRegistry(this.getRegistry(image));
        if (authConfigForRegistry != null) {
            return authConfigForRegistry;
        }
        return defaultAuthConfig;
    }

    private AuthConfig lookupAuthConfigForRegistry(String registry) {
        this.logger.debug("Looking up auth config for registry: " + registry);
        this.logger.debug("RegistryAuthLocator has configFile: " + this.configFile.getAbsolutePath() + " (" + (this.configFile.exists() ? "exists" : "does not exist") + ") and commandPathPrefix: " + this.commandPathPrefix);
        if (!this.configFile.isFile()) {
            return null;
        }
        try {
            Map config = this.objectMapper.readValue(this.configFile, Map.class);
            AuthConfig existingAuthConfig = this.findExistingAuthConfig(config, registry);
            if (existingAuthConfig != null) {
                return RegistryAuthLocator.decodeAuth(existingAuthConfig);
            }
            AuthConfig helperAuthConfig = this.authConfigUsingHelper(config, registry);
            if (helperAuthConfig != null) {
                return RegistryAuthLocator.decodeAuth(helperAuthConfig);
            }
            AuthConfig storeAuthConfig = this.authConfigUsingStore(config, registry);
            if (storeAuthConfig != null) {
                return RegistryAuthLocator.decodeAuth(storeAuthConfig);
            }
        }
        catch (Exception ex) {
            this.logger.error("Failure when attempting to lookup auth config (docker registry: {}, configFile: {}). Falling back to docker-java default behaviour", new Object[]{registry, this.configFile, ex});
        }
        return null;
    }

    public AuthConfigurations lookupAllAuthConfigs() {
        AuthConfigurations authConfigurations = new AuthConfigurations();
        this.logger.debug("RegistryAuthLocator has configFile: " + this.configFile.getAbsolutePath() + " (" + (this.configFile.exists() ? "exists" : "does not exist") + ") and commandPathPrefix: " + this.commandPathPrefix);
        if (!this.configFile.isFile()) {
            return authConfigurations;
        }
        try {
            HashSet registryAddresses = new HashSet();
            Map config = this.objectMapper.readValue(this.configFile, Map.class);
            Map authSectionRegistries = config.getOrDefault(AUTH_SECTION, new HashMap());
            this.logger.debug("Found registries in docker auths section: {}", authSectionRegistries.keySet());
            registryAddresses.addAll(authSectionRegistries.keySet());
            Map credHelperSectionRegistries = config.getOrDefault(HELPERS_SECTION, new HashMap());
            this.logger.debug("Found registries in docker credHelpers section: {}", credHelperSectionRegistries.keySet());
            registryAddresses.addAll(credHelperSectionRegistries.keySet());
            Object credStoreSection = config.get(CREDS_STORE_SECTION);
            if (credStoreSection instanceof String) {
                String credStoreCommand = this.commandPathPrefix + String.valueOf(credStoreSection) + this.helperSuffix;
                this.logger.debug("Executing docker credential helper: {} to locate auth configs", (Object)credStoreCommand);
                String credStoreResponse = this.runCommand(List.of(credStoreCommand, "list"));
                this.logger.debug("Credential helper response: {}", (Object)credStoreResponse);
                Map helperResponse = this.parseText(credStoreResponse, Map.class);
                if (helperResponse != null) {
                    this.logger.debug("Found registries in docker credential helper: {}", helperResponse.keySet());
                    registryAddresses.addAll(helperResponse.keySet());
                }
            }
            for (String registryAddress : registryAddresses) {
                AuthConfig registryAuthConfig = this.lookupAuthConfigForRegistry(registryAddress);
                if (registryAuthConfig == null) continue;
                authConfigurations.addConfig(registryAuthConfig);
            }
        }
        catch (Exception ex) {
            this.logger.error("Failure when attempting to lookup auth config (configFile: {}). Falling back to docker-java default behaviour", (Object)this.configFile, (Object)ex);
        }
        return authConfigurations;
    }

    public AuthConfigurations lookupAllAuthConfigs(DockerRegistryCredentials registryCredentials) {
        return this.lookupAllAuthConfigs(this.createAuthConfig(registryCredentials));
    }

    public AuthConfigurations lookupAllAuthConfigs(AuthConfig additionalAuthConfig) {
        AuthConfigurations allAuthConfigs = this.lookupAllAuthConfigs();
        if (this.isProvidedByBuild(additionalAuthConfig)) {
            allAuthConfigs.addConfig(additionalAuthConfig);
        }
        return allAuthConfigs;
    }

    private AuthConfig createAuthConfig(DockerRegistryCredentials registryCredentials) {
        AuthConfig authConfig = new AuthConfig();
        authConfig.withRegistryAddress((String)registryCredentials.getUrl().get());
        if (registryCredentials.getUsername().isPresent()) {
            authConfig.withUsername((String)registryCredentials.getUsername().get());
        }
        if (registryCredentials.getPassword().isPresent()) {
            authConfig.withPassword((String)registryCredentials.getPassword().get());
        }
        if (registryCredentials.getEmail().isPresent()) {
            authConfig.withEmail((String)registryCredentials.getEmail().get());
        }
        return authConfig;
    }

    private static String configLocation() {
        String defaultDir = System.getProperty(USER_HOME) + File.separator + DOCKER_DIR;
        String dir = System.getenv().getOrDefault(DOCKER_CONFIG, defaultDir);
        return dir + File.separator + CONFIG_JSON;
    }

    public String getRegistry(String image) {
        NameParser.ReposTag tag = NameParser.parseRepositoryTag(image);
        NameParser.HostnameReposName repository = NameParser.resolveRepositoryName(tag.repos);
        return repository.hostname;
    }

    private AuthConfig findExistingAuthConfig(Map<String, Object> config, String repository) {
        Map authMap;
        Map.Entry<String, Object> entry = RegistryAuthLocator.findAuthNode(config, repository);
        if (entry != null && entry.getValue() != null && entry.getValue() instanceof Map && (authMap = (Map)entry.getValue()).size() > 0) {
            String authJson;
            try {
                authJson = this.objectMapper.writeValueAsString(entry.getValue());
            }
            catch (JsonProcessingException e) {
                throw new UncheckedIOException(e);
            }
            AuthConfig authCfg = this.parseText(authJson, AuthConfig.class);
            if (authCfg == null) {
                return null;
            }
            return authCfg.withRegistryAddress(entry.getKey());
        }
        this.logger.debug("No existing AuthConfig found");
        return null;
    }

    private static Map.Entry<String, Object> findAuthNode(Map<String, Object> config, String repository) {
        Map auths = (Map)config.get(AUTH_SECTION);
        if (auths != null && auths.size() > 0) {
            for (Map.Entry<String, Object> entry : auths.entrySet()) {
                if (!((String)entry.getKey()).endsWith("://" + repository) && !((String)entry.getKey()).equals(repository)) continue;
                return entry;
            }
        }
        return null;
    }

    private AuthConfig authConfigUsingHelper(Map<String, Object> config, String repository) {
        Object helperNode;
        Map credHelpers = (Map)config.get(HELPERS_SECTION);
        if (credHelpers != null && credHelpers.size() > 0 && (helperNode = credHelpers.get(repository)) instanceof String) {
            String helper = (String)helperNode;
            return this.runCredentialProvider(repository, helper);
        }
        this.logger.debug("No helper found in the {} section", (Object)HELPERS_SECTION);
        return null;
    }

    private AuthConfig runCredentialProvider(String hostName, String credHelper) {
        String credentialHelperName = this.commandPathPrefix + credHelper + this.helperSuffix;
        this.logger.debug("Executing docker credential helper: {} to locate auth config for: {}", (Object)credentialHelperName, (Object)hostName);
        String data = this.runCommand(List.of(credentialHelperName, "get"), hostName);
        this.logger.debug("Credential helper response: {}", (Object)data);
        Map helperResponse = this.parseText(data, Map.class);
        if (helperResponse == null) {
            return null;
        }
        this.logger.debug("Credential helper provided auth config for: {}", (Object)hostName);
        String registryAddress = helperResponse.get("ServerURL") != null ? (String)helperResponse.get("ServerURL") : hostName;
        return new AuthConfig().withRegistryAddress(registryAddress).withUsername((String)helperResponse.get("Username")).withPassword((String)helperResponse.get("Secret"));
    }

    private AuthConfig authConfigUsingStore(Map<String, Object> config, String repository) {
        Object credsStoreNode = config.get(CREDS_STORE_SECTION);
        if (credsStoreNode instanceof String) {
            String credsStore = (String)credsStoreNode;
            return this.runCredentialProvider(repository, credsStore);
        }
        this.logger.debug("No helper found in the {} section", (Object)CREDS_STORE_SECTION);
        return null;
    }

    private String runCommand(List<String> command) {
        return this.runCommand(command, null);
    }

    private String runCommand(List<String> command, String input) {
        try {
            ByteArrayOutputStream sOut = new ByteArrayOutputStream();
            ByteArrayOutputStream eOut = new ByteArrayOutputStream();
            this.execOperations.exec(execSpec -> {
                execSpec.setCommandLine(command);
                execSpec.setStandardOutput((OutputStream)sOut);
                execSpec.setErrorOutput((OutputStream)eOut);
                execSpec.setIgnoreExitValue(true);
                if (input != null) {
                    execSpec.setStandardInput((InputStream)new ByteArrayInputStream(input.getBytes()));
                }
            });
            if (eOut.size() > 0) {
                this.logger.error("{}: {}", command, (Object)eOut.toString());
            }
            return sOut.toString();
        }
        catch (Exception e) {
            this.logger.error("Failure running command ({})", command);
            throw e;
        }
    }

    private static AuthConfig decodeAuth(AuthConfig config) {
        if (config.getAuth() == null) {
            return config;
        }
        String str = new String(Base64.getDecoder().decode(config.getAuth()), StandardCharsets.UTF_8);
        String[] parts = str.split(":", 2);
        if (parts.length != 2) {
            throw new RuntimeException("Invalid auth configuration file");
        }
        config.withUsername(parts[0]);
        config.withPassword(parts[1]);
        config.withAuth(null);
        return config;
    }

    private <T> T parseText(String data, Class<T> valueType) {
        try {
            return this.objectMapper.readValue(data, valueType);
        }
        catch (Exception e) {
            this.logger.debug("Failure parsing the json response {}", (Object)data, (Object)e);
            return null;
        }
    }

    void setLogger(Logger logger) {
        this.logger = logger;
    }

    public static class Factory {
        private final ExecOperations execOperations;

        @Inject
        public Factory(ExecOperations execOperations) {
            this.execOperations = execOperations;
        }

        RegistryAuthLocator withConfigAndCommandPathPrefix(File configFile, String commandPathPrefix, String helperSuffix) {
            return new RegistryAuthLocator(this.execOperations, configFile, commandPathPrefix, helperSuffix);
        }

        RegistryAuthLocator withConfig(File configFile) {
            return new RegistryAuthLocator(this.execOperations, configFile);
        }

        public RegistryAuthLocator withDefaults() {
            return new RegistryAuthLocator(this.execOperations);
        }
    }
}

