package com.atlassian.event.internal;

import com.atlassian.event.spi.ListenerInvoker;
import com.google.common.collect.Sets;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Optional;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A listener invoker that knows how to call a given single parameter method on a given object.
 *
 * @since 2.0
 */
final class SingleParameterMethodListenerInvoker implements ListenerInvoker {
    private final Method method;
    private final Object listener;
    private final Optional<String> scope;


    public SingleParameterMethodListenerInvoker(Object listener, Method method) {
        this(listener, method, Optional.empty());
    }

    public SingleParameterMethodListenerInvoker(Object listener, Method method, Optional<String> scope) {
        this.listener = checkNotNull(listener);
        this.method = checkNotNull(method);
        this.scope = checkNotNull(scope);
    }

    public Set<Class<?>> getSupportedEventTypes() {
        return Sets.newHashSet(method.getParameterTypes());
    }

    public void invoke(Object event) {
        try {
            method.invoke(listener, event);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Listener: " + listener.getClass().getName()
                    + " event: " + event.getClass().getName(), e);
        } catch (InvocationTargetException e) {
            if (e.getCause() == null) {
                throw new RuntimeException("Listener: " + listener.getClass().getName()
                        + " event: " + event.getClass().getName(), e);
            } else if (e.getCause().getMessage() == null) {
                throw new RuntimeException("Listener: " + listener.getClass().getName()
                        + " event: " + event.getClass().getName(), e.getCause());
            } else {
                throw new RuntimeException(e.getCause().getMessage()
                        + ". Listener: " + listener.getClass().getName()
                        + " event: " + event.getClass().getName(),
                        e.getCause());
            }
        }
    }

    public boolean supportAsynchronousEvents() {
        return true;
    }

    @Override
    public Optional<String> getScope()
    {
        return scope;
    }

    @Override
    public String toString() {
        return "SingleParameterMethodListenerInvoker{method=" + method + ", listener=" + paranoidToString(listener) + '}';
    }

    /**
     * Calls the object's toString() method. If an exception is thrown from the object's toString() method then this
     * method falls back to the "identity" toString() (as per JVM defaults).
     *
     * @param object an Object
     * @return a human-readable String representation of the object
     */
    private String paranoidToString(Object object) {
        try {
            return String.valueOf(object);
        } catch (RuntimeException e) {
            return object.getClass().getName() + '@' + Integer.toHexString(System.identityHashCode(object));
        }
    }
}
