/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bamboo.utils;

import com.atlassian.bamboo.utils.BambooFiles;
import com.atlassian.bamboo.utils.concurrent.AtomicUtils;
import com.google.common.base.Preconditions;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.util.EnumSet;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.VisibleForTesting;

public class BambooPathUtils {
    private static final Logger log = LogManager.getLogger(BambooPathUtils.class);
    private static EnumSet<PosixFilePermission> ALL_PERMISSIONS = EnumSet.allOf(PosixFilePermission.class);

    public static boolean deleteQuietly(@Nullable Path path) {
        if (path == null) {
            return false;
        }
        if (path.toFile().isDirectory()) {
            try {
                BambooPathUtils.cleanDirectory(path);
            }
            catch (Exception e) {
                log.debug("Error cleaning {}", (Object)path, (Object)e);
            }
        }
        try {
            BambooFiles.delete(path);
            return true;
        }
        catch (Exception e) {
            log.debug("Error deleting {}", (Object)path, (Object)e);
            return false;
        }
    }

    public static void cleanDirectory(@NotNull Path directoryToCleanOrSymlink) throws IOException {
        Preconditions.checkArgument((boolean)directoryToCleanOrSymlink.toFile().exists(), (Object)(String.valueOf(directoryToCleanOrSymlink) + " does not exist"));
        Preconditions.checkArgument((boolean)directoryToCleanOrSymlink.toFile().isDirectory(), (Object)(String.valueOf(directoryToCleanOrSymlink) + " is not a directory"));
        Path directoryToClean = BambooPathUtils.toRealPathIfNeeded(directoryToCleanOrSymlink);
        AtomicReference<IOException> exceptionRef = new AtomicReference<IOException>();
        boolean tryHarder = false;
        Files.walkFileTree(directoryToClean, BambooPathUtils.newDeletingVisitor(directoryToClean, exceptionRef, tryHarder));
        if (exceptionRef.get() instanceof DirectoryNotEmptyException) {
            exceptionRef.set(null);
            tryHarder = true;
            Files.walkFileTree(directoryToClean, BambooPathUtils.newDeletingVisitor(directoryToClean, exceptionRef, tryHarder));
        }
        BambooPathUtils.handleCleanDirectoryExceptions(exceptionRef, directoryToClean);
    }

    private static void handleCleanDirectoryExceptions(@NotNull AtomicReference<IOException> exceptionRef, @NotNull Path directoryToClean) throws IOException {
        try {
            IOException exception = exceptionRef.get();
            if (exception != null) {
                if (!(exception instanceof NoSuchFileException)) {
                    throw exception;
                }
                log.debug("Error while attempting to clean directory {}", (Object)directoryToClean, (Object)exception);
            }
            if (Objects.requireNonNull(directoryToClean.toFile().list()).length > 0) {
                throw new IOException("Directory not empty after cleaning attempt");
            }
        }
        catch (RuntimeException e) {
            throw new IOException("An unexpected error occurred while verifying the directory cleaning", e);
        }
    }

    @NotNull
    private static SimpleFileVisitor<Path> newDeletingVisitor(final Path directoryToClean, final AtomicReference<IOException> exceptionRef, final boolean shouldTryHarder) {
        return new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                BambooPathUtils.deleteNoThrow(file, exceptionRef);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) {
                AtomicUtils.setIfNotNull(exceptionRef, exc);
                BambooPathUtils.deleteNoThrow(file, exceptionRef);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                PosixFileAttributeView fileAttributeView;
                if (shouldTryHarder && !dir.equals(directoryToClean) && (fileAttributeView = Files.getFileAttributeView(dir, PosixFileAttributeView.class, new LinkOption[0])) != null) {
                    fileAttributeView.setPermissions(ALL_PERMISSIONS);
                }
                return super.preVisitDirectory(dir, attrs);
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                AtomicUtils.setIfNotNull(exceptionRef, exc);
                if (!dir.equals(directoryToClean)) {
                    BambooPathUtils.deleteNoThrow(dir, exceptionRef);
                }
                return FileVisitResult.CONTINUE;
            }
        };
    }

    public static void deleteDirectory(@NotNull Path path) throws IOException {
        if (!path.toFile().exists()) {
            return;
        }
        if (!Files.isSymbolicLink(path)) {
            BambooPathUtils.cleanDirectory(path);
        }
        BambooFiles.delete(path);
    }

    public static void forceDelete(@NotNull Path path) throws IOException {
        if (path.toFile().isDirectory()) {
            BambooPathUtils.deleteDirectory(path);
        } else {
            try {
                BambooFiles.delete(path);
            }
            catch (NoSuchFileException e) {
                throw new FileNotFoundException(e.getMessage());
            }
        }
    }

    @VisibleForTesting
    protected static void deleteNoThrow(Path file, AtomicReference<IOException> exceptionRef) {
        try {
            BambooFiles.delete(file);
        }
        catch (IOException e) {
            exceptionRef.set(e);
        }
    }

    public static Path toRealPathIfNeeded(@NotNull Path maybeSymlink) throws IOException {
        return Files.isSymbolicLink(maybeSymlink) ? maybeSymlink.toRealPath(new LinkOption[0]) : maybeSymlink;
    }

    public static void writeStringToFile(Path file, String data, Charset charset) throws IOException {
        try (BufferedWriter writer = Files.newBufferedWriter(file, charset, new OpenOption[0]);){
            IOUtils.write((String)data, (Writer)writer);
        }
    }

    @Nullable
    public static Path toPath(@Nullable File file) {
        if (file == null) {
            return null;
        }
        return file.toPath();
    }

    public static OutputStream openOutputStream(Path file) throws IOException {
        return BambooPathUtils.openOutputStream(file, false);
    }

    public static OutputStream openOutputStream(Path file, boolean append) throws IOException {
        try {
            return BambooPathUtils.openOutputStreamInternal(file, append);
        }
        catch (IOException iOException) {
            if (file.toFile().exists()) {
                if (file.toFile().isDirectory()) {
                    throw new IOException("File '" + String.valueOf(file) + "' exists but is a directory");
                }
                if (!Files.isWritable(file)) {
                    throw new IOException("File '" + String.valueOf(file) + "' cannot be written to");
                }
            } else {
                Path parent = file.getParent();
                if (parent != null && !BambooPathUtils.mkdirs(parent) && !parent.toFile().isDirectory()) {
                    throw new IOException("Directory '" + String.valueOf(parent) + "' could not be created");
                }
            }
            return BambooPathUtils.openOutputStreamInternal(file, append);
        }
    }

    public static boolean contentEquals(@NotNull Path file1, @NotNull Path file2) throws IOException {
        boolean file1Exists = file1.toFile().exists();
        if (file1Exists != file2.toFile().exists()) {
            return false;
        }
        if (!file1Exists) {
            return true;
        }
        if (file1.toFile().isDirectory() || file2.toFile().isDirectory()) {
            throw new IOException("Can't compare directories, only files");
        }
        if (Files.size(file1) != Files.size(file2)) {
            return false;
        }
        if (file1.toRealPath(new LinkOption[0]).equals(file2.toRealPath(new LinkOption[0]))) {
            return true;
        }
        try (InputStream input1 = Files.newInputStream(file1, new OpenOption[0]);){
            boolean bl;
            block17: {
                InputStream input2 = Files.newInputStream(file2, new OpenOption[0]);
                try {
                    bl = IOUtils.contentEquals((InputStream)input1, (InputStream)input2);
                    if (input2 == null) break block17;
                }
                catch (Throwable throwable) {
                    if (input2 != null) {
                        try {
                            input2.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                input2.close();
            }
            return bl;
        }
    }

    public static String readFileToString(Path path, Charset encoding) throws IOException {
        return new String(Files.readAllBytes(path), encoding);
    }

    private static OutputStream openOutputStreamInternal(Path file, boolean append) throws IOException {
        return append ? Files.newOutputStream(file, StandardOpenOption.CREATE, StandardOpenOption.APPEND, StandardOpenOption.WRITE) : Files.newOutputStream(file, new OpenOption[0]);
    }

    private static boolean mkdirs(Path parent) {
        try {
            Files.createDirectories(parent, new FileAttribute[0]);
            return true;
        }
        catch (IOException ignored) {
            return false;
        }
    }
}

