package com.xebialabs.deployit.core.serializers

import com.fasterxml.jackson.annotation.SimpleObjectIdResolver
import com.fasterxml.jackson.core.{JsonParser, JsonToken}
import com.fasterxml.jackson.databind.ser.impl.ObjectIdWriter
import com.fasterxml.jackson.databind.{DeserializationContext, JsonDeserializer}
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.util.PasswordEncrypter

import java.util
import scala.jdk.CollectionConverters._

class ConfigurationItemDeserializer(passwordEncrypter: PasswordEncrypter, ciIdWriter: ObjectIdWriter) extends JsonDeserializer[ConfigurationItem] {

  override def deserialize(p: JsonParser, ctxt: DeserializationContext): ConfigurationItem = {
    if (p.isExpectedStartObjectToken) {
      val id = readNextString(p)
      val ciType = readNextString(p)
      val descriptor = Type.valueOf(ciType).getDescriptor
      val ci: ConfigurationItem = descriptor.newInstance(id)
      findObjectId(ctxt, id).bindItem(ci)
      while (p.nextToken() != JsonToken.END_OBJECT) {
        Option(descriptor.getPropertyDescriptor(p.getCurrentName)).foreach { pd =>
          p.nextToken()
          pd.getKind match {
            case BOOLEAN =>
              pd.set(ci, p.getValueAsBoolean)
            case CI =>
              pd.set(ci, ctxt.readValue(p, classOf[ConfigurationItem]))
            case DATE =>
              pd.set(ci, ctxt.readValue(p, classOf[util.Date]))
            case STRING =>
              pd.set(ci, readStringValue(p, pd.isPassword))
            case ENUM =>
              pd.set(ci, p.getValueAsString)
            case INTEGER =>
              pd.set(ci, p.getValueAsInt)
            case LIST_OF_CI =>
              pd.set(ci, new util.ArrayList(readCis(p, ctxt)))
            case LIST_OF_STRING =>
              pd.set(ci, convertList(new util.ArrayList(readStrings(p, pd.isPassword)), pd.isPassword, passwordEncrypter))
            case MAP_STRING_STRING =>
              pd.set(ci, convertMap(readMap(p, pd.isPassword), pd.isPassword, passwordEncrypter))
            case SET_OF_CI =>
              pd.set(ci, new util.HashSet(readCis(p, ctxt)))
            case SET_OF_STRING =>
              pd.set(ci, convertSet(new util.HashSet(readStrings(p, pd.isPassword)), pd.isPassword, passwordEncrypter))
          }
        }
      }
      ci
    } else {
      val id = p.readValueAs(classOf[String])
      findObjectId(ctxt, id).resolve().asInstanceOf[ConfigurationItem]
    }
  }

  private def findObjectId(ctxt: DeserializationContext, id: String) =
    ctxt.findObjectId(id, ConfigurationItemObjectIdGenerator, new SimpleObjectIdResolver)

  private def readCis(p: JsonParser, ctxt: DeserializationContext): util.Collection[ConfigurationItem] =
    Iterator.continually(p.nextToken()).takeWhile(_ != JsonToken.END_ARRAY).map(_ => ctxt.readValue(p, classOf[ConfigurationItem])).toSeq.asJava

  private def readStrings(p: JsonParser, isPassword: Boolean): util.Collection[String] =
    Iterator.continually(p.nextToken()).takeWhile(_ != JsonToken.END_ARRAY).map(_ => readStringValue(p, isPassword)).toSeq.asJava

  private def readMap(p: JsonParser, isPassword: Boolean): util.Map[String, String] = {
    val result = new util.LinkedHashMap[String, String]
    while ( {
      val token = p.nextToken(); token != JsonToken.END_OBJECT && token != JsonToken.END_ARRAY
    }) {
      val key = p.getCurrentName
      p.nextToken()
      val value = readStringValue(p, isPassword)
      result.put(key, value)
    }
    result
  }

  private def readStringValue(p: JsonParser, isPassword: Boolean): String = {
    var value = p.getValueAsString
    if (isPassword) value = passwordEncrypter.ensureDecrypted(value)
    value
  }

  private def readNextString(p: JsonParser) = {
    p.nextToken()
    p.getCurrentName
    p.nextToken()
    p.getValueAsString
  }
}
