package com.xebialabs.deployit.core.serializers

import java.util
import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.databind.ser.impl.ObjectIdWriter
import com.fasterxml.jackson.databind.{JsonSerializer, SerializerProvider}
import com.xebialabs.deployit.core.util.CollectionUtil
import com.xebialabs.deployit.core.{ListOfStringView, MapStringStringView, SetOfStringView, StringValue}
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.StringValueConverter.valueToString
import com.xebialabs.deployit.util.PasswordEncrypter


class ConfigurationItemSerializer(passwordEncrypter: PasswordEncrypter, ciIdWriter: ObjectIdWriter) extends JsonSerializer[ConfigurationItem] {
  override def serialize(ci: ConfigurationItem, gen: JsonGenerator, serializers: SerializerProvider): Unit = {
    def writeCi(ci: ConfigurationItem)(properties: => Unit):Unit = {
      val objectId = serializers.findObjectId(ci, ConfigurationItemObjectIdGenerator)
      if (!objectId.writeAsId(gen, serializers, ciIdWriter)) {
        gen.writeStartObject()
        objectId.generateId(ci)
        objectId.writeAsField(gen, serializers, ciIdWriter)
        gen.writeStringField("type", ci.getType.toString)
        properties
        gen.writeEndObject()
      }
    }
    def writeObject(value: AnyRef): Unit = {
      val ser = serializers.findValueSerializer(value.getClass)
      ser.serialize(value, gen, serializers)
    }
    writeCi(ci) {
      forAllNonTransientProperties(ci.getType) { pd =>
        getValue(ci, pd).foreach { value =>
          pd.getKind match {
            case BOOLEAN =>
              gen.writeBooleanField(pd.getName, value.asInstanceOf[Boolean])
            case CI =>
              value match {
                case null => //ignore
                case ref: ConfigurationItem =>
                  gen.writeFieldName(pd.getName)
                  writeCi(ref) {}
                case _ =>
                  throw new IllegalStateException("Cannot serialize CI type property not referencing CI.")
              }
            case DATE =>
              gen.writeFieldName(pd.getName)
              writeObject(value)
            case ENUM | STRING =>
              gen.writeStringField(pd.getName, value.toString)
            case INTEGER =>
              gen.writeNumberField(pd.getName, value.asInstanceOf[Number].longValue())
            case LIST_OF_CI | SET_OF_CI =>
              val collection = value.asInstanceOf[util.Collection[ConfigurationItem]]
              if (!collection.isEmpty) {
                gen.writeFieldName(pd.getName)
                gen.writeStartArray()
                collection.forEach(writeObject(_))
                gen.writeEndArray()
              }
            case LIST_OF_STRING | SET_OF_STRING =>
              val collection = value.asInstanceOf[util.Collection[String]]
              if (!collection.isEmpty) {
                gen.writeFieldName(pd.getName)
                gen.writeStartArray()
                collection.forEach(gen.writeString(_))
                gen.writeEndArray()
              }
            case MAP_STRING_STRING =>
              val map = value.asInstanceOf[util.Map[String, String]]
              if (!map.isEmpty) {
                gen.writeFieldName(pd.getName)
                gen.writeStartObject()
                map.entrySet().forEach(entry => gen.writeStringField(entry.getKey, entry.getValue))
                gen.writeEndObject()
              }
          }
        }
      }
    }
  }

  private def getValue(ci: ConfigurationItem, pd: PropertyDescriptor): Option[AnyRef] = {
    Option(pd.get(ci)).map { value =>
      pd.getKind match {
        case PropertyKind.STRING if pd.isPassword =>
          passwordEncrypter.ensureEncrypted(value.toString)
        case PropertyKind.SET_OF_STRING =>
          var set = value match {
            case set: SetOfStringView => set
            case _ => SetOfStringView.from(value.asInstanceOf[util.Set[String]])
          }
          if (pd.isPassword)
            set = set.encrypt
          CollectionUtil.apply[StringValue, String](set.getWrapped, valueToString(passwordEncrypter))
        case PropertyKind.LIST_OF_STRING =>
          var list = value match {
            case list: ListOfStringView => list
            case _ => ListOfStringView.from(value.asInstanceOf[util.List[String]])
          }
          if (pd.isPassword)
            list = list.encrypt
          CollectionUtil.apply[StringValue, String](list.getWrapped, valueToString(passwordEncrypter))
        case PropertyKind.MAP_STRING_STRING =>
          var map = value match {
            case map: MapStringStringView => map
            case _ => MapStringStringView.from(value.asInstanceOf[util.Map[String, String]])
          }
          if (pd.isPassword)
            map = map.encrypt
          CollectionUtil.apply[StringValue, String](map.getWrapped, valueToString(passwordEncrypter))
        case _ => value
      }
    }
  }
}
