package com.xebialabs.deployit.hocon

import com.typesafe.config.{ConfigResolveOptions, ConfigResolver, ConfigValue}
import com.xebialabs.deployit.hocon.HoconPropertySourceLoader.{BASIC_NAME_SUFFIX, WITH_DEFAULTS_NAME_SUFFIX}
import org.springframework.boot.env.PropertySourceLoader
import org.springframework.core.env.{MapPropertySource, PropertySource}
import org.springframework.core.io.Resource

import java.io.IOException
import java.util
import java.util.Collections.{emptyList, singletonMap}

object HoconPropertySourceLoader {
  val BASIC_NAME_SUFFIX: String = " (basic only)"
  val WITH_DEFAULTS_NAME_SUFFIX: String = " (with default)"

  private var instance: Option[HoconConfigLoader] = None

  private lazy val defaultHoconConfigLoader: HoconConfigLoader = new ServerHoconConfigLoader

  def setInstance(hoconConfigLoader: HoconConfigLoader): Unit = {
    instance = Option(hoconConfigLoader)
  }

  def getInstance: HoconConfigLoader =
    instance.getOrElse(defaultHoconConfigLoader)
}

class HoconPropertySourceLoader extends PropertySourceLoader {
  /**
   * Returns the file extensions that the loader supports (excluding the '.').
   *
   * @return the file extensions
   */
  override def getFileExtensions: Array[String] = Array[String](xs = "conf")

  /**
   * Load the resource into one or more property sources. Implementations may either
   * return a list containing a single source, or in the case of a multi-document format
   * such as yaml a source for each document in the resource.
   *
   * @param name     the root name of the property source. If multiple documents are loaded
   *                 an additional suffix should be added to the name for each source loaded.
   * @param resource the resource to load
   * @return a list property sources
   * @throws IOException if the source cannot be loaded
   */
  @throws[IOException]
  override def load(name: String, resource: Resource): util.List[PropertySource[_]] = {

    val config = HoconPropertySourceLoader.getInstance.loadConfig(resource)

    val result: util.Map[String, AnyRef] = new util.LinkedHashMap[String, AnyRef]
    buildFlattenedMap(result, config.root().unwrapped(), root = null)
    if (result.isEmpty) {
      return emptyList
    }

    val basicConfig = HoconPropertySourceLoader.getInstance.loadBasicConfig(resource)

    val basicResult: util.Map[String, AnyRef] = new util.LinkedHashMap[String, AnyRef]
    buildFlattenedMap(basicResult, basicConfig.resolve(ConfigResolveOptions.defaults().appendResolver(new ConfigResolver {

      override def lookup(path: String): ConfigValue = config.getValue(path)

      override def withFallback(fallback: ConfigResolver): ConfigResolver = fallback
    })).root().unwrapped(), root = null)

    util.List.of(
      new MapPropertySource(name + BASIC_NAME_SUFFIX, basicResult),
      new MapPropertySource(name + WITH_DEFAULTS_NAME_SUFFIX, result)
    )
  }

  private def buildFlattenedMap(result: util.Map[String, AnyRef], source: util.Map[String, AnyRef], root: String): Unit = {
    val rootHasText: Boolean = null != root && root.trim.nonEmpty

    source.forEach((key: String, value: AnyRef) => {
      def foo(key: String, value: AnyRef): Any = {
        var path: String = null
        if (rootHasText) {
          if (key.startsWith("[")) {
            path = root + key
          }
          else if (key.contains(".")) {
            path = root + ".\"" + key + "\""
          }
          else {
            path = root + "." + key
          }
        }
        else {
          path = key
        }
        if (value.isInstanceOf[util.Map[_, _]]) {
          @SuppressWarnings(Array("unchecked")) val map: util.Map[String, AnyRef] = value.asInstanceOf[util.Map[String, AnyRef]]
          buildFlattenedMap(result, map, path)
        }
        else {
          if (value.isInstanceOf[util.Collection[_]]) {
            @SuppressWarnings(Array("unchecked")) val collection: util.Collection[AnyRef] = value.asInstanceOf[util.Collection[AnyRef]]
            var count = 0

            collection.forEach(item => {
              buildFlattenedMap(result, singletonMap("[" + {
                count
              } + "]", item), path)
              count += 1
            })

          }
          else {
            result.put(path, if (null == value) "" else value)
          }
        }
      }

      foo(key, value)
    })
  }
}
