package com.xebialabs.deployit.env

import org.springframework.context.annotation.Configuration
import org.springframework.context.{ApplicationContextInitializer, ConfigurableApplicationContext}
import org.springframework.core.Ordered
import org.springframework.core.env.{EnumerablePropertySource, Environment}

import java.util.concurrent.TimeUnit

@Configuration(proxyBeanMethods = false)
class ServerDynamicPropertySourceConfiguration extends ApplicationContextInitializer[ConfigurableApplicationContext] with Ordered {

  // it needs to be after org.springframework.cloud.bootstrap.config.PropertySourceBootstrapConfiguration
  override def getOrder: Int = Ordered.HIGHEST_PRECEDENCE + 11

  class DynamicPropertySource(mappings: Map[String, () => String]) extends EnumerablePropertySource[Map[String, () => String]]("server-dynamic-env", mappings) {

    override def getProperty(name: String): AnyRef =
      mappings.get(name)
        .map(_ ())
        .orNull

    override def containsProperty(name: String): Boolean =
      mappings.contains(name)

    override def getPropertyNames: Array[String] = mappings.keys.toArray
  }

  override def initialize(applicationContext: ConfigurableApplicationContext): Unit = {
    val environment = applicationContext.getEnvironment
    val propertySources = environment.getPropertySources
    propertySources.addFirst(new DynamicPropertySource(createMappings(environment)))
  }

  private def createMappings(environment: Environment) = Map[String, () => String](
    "spring.session.timeout" -> { () =>
      val timeout = environment.getProperty("deploy.client.session.timeout-minute", "0").toInt
      if (timeout > 0)
        TimeUnit.MINUTES.toSeconds(timeout).toString
      else
        TimeUnit.DAYS.toSeconds(30).toString
    }
  )
}
