package com.xebialabs.deployit.yaml

import org.springframework.boot.env.{OriginTrackedMapPropertySource, PropertySourceLoader}
import org.springframework.core.env.PropertySource
import org.springframework.core.io.Resource
import org.springframework.util.ClassUtils

import java.util
import java.util.Collections
import scala.jdk.CollectionConverters._

/**
 * Allow to map null value from yaml. Spring default behaviour is to use empty string in case of null value from yaml file.
 * Related issue: https://github.com/spring-projects/spring-framework/issues/19986
 */
class YamlPropertySourceLoader extends PropertySourceLoader {

  override def getFileExtensions: Array[String] = Array[String]("yml", "yaml")

  override def load(name: String, resource: Resource): util.List[PropertySource[_]] = {
    if (!ClassUtils.isPresent("org.yaml.snakeyaml.Yaml", null)) {
      throw new IllegalStateException("Attempted to load " + name + " but snakeyaml was not found on the classpath")
    }
    val loaded: List[util.Map[String, AnyRef]] = new CustomOriginTrackedYamlLoader(resource).load.asScala.toList
    if (loaded.isEmpty) {
      return Collections.emptyList
    }
    val propertySources = new util.ArrayList[PropertySource[_]](loaded.size)
    loaded.zipWithIndex.foreach { case (loadedEntry, index) =>
      val documentNumber = if (loaded.size != 1)
        " (document #" + index + ")"
      else
        ""
      propertySources.add(new OriginTrackedMapPropertySource(name + documentNumber, Collections.unmodifiableMap(loadedEntry), true))
    }
    propertySources
  }
}