package com.xebialabs.deployit.core.config

import grizzled.slf4j.Logging
import org.springframework.core.env.{ConfigurableEnvironment, EnumerablePropertySource, Environment, MutablePropertySources}
import org.springframework.stereotype.Component

import java.util
import java.util.{Properties, Map => JMap}
import scala.util.matching.Regex

@Component
class SpringConfigurationCollector(private val environment: Environment) extends Logging {

  def environmentProperties(filter: String => Boolean): (Properties, JMap[String, AnyRef]) = {
    val originalMap = environmentMap(filter)
    val properties = new Properties()
    val map = new util.HashMap[String, AnyRef]()
    originalMap.entrySet().forEach { entry =>
      if (entry.getKey.contains("\"")) {
        map.put(entry.getKey, entry.getValue)
      } else {
        properties.put(entry.getKey, entry.getValue.toString)
      }
    }
    (properties, map)
  }

  private def environmentMap(filter: String => Boolean): JMap[String, AnyRef] = {
    val resultMap = new util.HashMap[String, AnyRef]
    propertySources.iterator()
      .forEachRemaining {
        case source@(enumerablePropertySource: EnumerablePropertySource[_]) if filter(enumerablePropertySource.getName) =>
          enumerablePropertySource.getPropertyNames
            .foreach { property =>
              val transformedProperty = transformPropertyKey(property)
              resultMap.computeIfAbsent(transformedProperty, _ => source.getProperty(property))
            }
        case otherSource =>
          logger.debug(s"Only EnumerablePropertySource are supported. ${otherSource.getClass} type is not supported with name ${otherSource.getName}")
      }
    resultMap
  }

  private def propertySources: MutablePropertySources = this.environment.asInstanceOf[ConfigurableEnvironment].getPropertySources

  private val numberPatternOnEnd: Regex = "\\[([0-9]+)\\]$".r
  private val numberPatternInMiddle: Regex = "\\[([0-9]+)\\]\\.".r

  private def transformPropertyKey(key: String): String = {
    val result = numberPatternOnEnd.replaceAllIn(key, matching => s".${matching.group(1)}")
    numberPatternInMiddle.replaceAllIn(result, matching => s".${matching.group(1)}.")
  }
}
