package com.xebialabs.deployit.core.config.session;


import com.xebialabs.deployit.core.sql.spring.DeployJdbcTemplate;
import com.xebialabs.deployit.repository.sql.DeployJdbcIndexedSessionRepository;
import com.xebialabs.deployit.security.SpringSessionConverterFactory;
import org.springframework.beans.factory.BeanClassLoaderAware;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.session.JdbcSessionProperties;
import org.springframework.boot.autoconfigure.session.SessionProperties;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.EmbeddedValueResolverAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.ImportAware;
import org.springframework.context.annotation.Primary;
import org.springframework.core.annotation.AnnotationAttributes;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.support.GenericConversionService;
import org.springframework.core.serializer.support.DeserializingConverter;
import org.springframework.core.serializer.support.SerializingConverter;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.jdbc.support.MetaDataAccessException;
import org.springframework.jdbc.support.lob.DefaultLobHandler;
import org.springframework.jdbc.support.lob.LobHandler;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.security.core.session.SessionRegistry;
import org.springframework.session.*;
import org.springframework.session.config.SessionRepositoryCustomizer;
import org.springframework.session.config.annotation.web.http.SpringHttpSessionConfiguration;
import org.springframework.session.jdbc.JdbcIndexedSessionRepository;
import org.springframework.session.jdbc.config.annotation.SpringSessionDataSource;
import org.springframework.session.jdbc.config.annotation.web.http.EnableJdbcHttpSession;
import org.springframework.session.security.SpringSessionBackedSessionRegistry;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionOperations;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.StringUtils;
import org.springframework.util.StringValueResolver;

import javax.sql.DataSource;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@ConditionalOnClass({JdbcTemplate.class, DeployJdbcIndexedSessionRepository.class})
@ConditionalOnBean(DataSource.class)
@EnableConfigurationProperties(JdbcSessionProperties.class)
@ConditionalOnMissingBean(SessionRepository.class)
public class SpringJdbcSessionConfig extends SpringHttpSessionConfiguration
        implements BeanClassLoaderAware, EmbeddedValueResolverAware, ImportAware {

    static final String DEFAULT_CLEANUP_CRON = "0 * * * * *";

    private Integer maxInactiveIntervalInSeconds = MapSession.DEFAULT_MAX_INACTIVE_INTERVAL_SECONDS;

    private String tableName = JdbcIndexedSessionRepository.DEFAULT_TABLE_NAME;

    private String cleanupCron = DEFAULT_CLEANUP_CRON;

    private FlushMode flushMode = FlushMode.ON_SAVE;

    private SaveMode saveMode = SaveMode.ON_SET_ATTRIBUTE;

    private DataSource dataSource;

    private PlatformTransactionManager transactionManager;

    private TransactionOperations transactionOperations;

    private IndexResolver<Session> indexResolver;

    private LobHandler lobHandler;

    private ConversionService springSessionConversionService;

    private ConversionService conversionService;

    private List<SessionRepositoryCustomizer<DeployJdbcIndexedSessionRepository>> sessionRepositoryCustomizers;

    private ClassLoader classLoader;

    private StringValueResolver embeddedValueResolver;

    private SpringSessionConverterFactory springSessionConverterFactory;

    @Bean
    @Primary
    public DeployJdbcIndexedSessionRepository sessionRepository() {
        JdbcTemplate jdbcTemplate = createJdbcTemplate(this.dataSource);
        if (this.transactionOperations == null) {
            this.transactionOperations = createTransactionTemplate(this.transactionManager);
        }
        DeployJdbcIndexedSessionRepository sessionRepository = new DeployJdbcIndexedSessionRepository(jdbcTemplate,
                this.transactionOperations);
        if (StringUtils.hasText(this.tableName)) {
            sessionRepository.setTableName(this.tableName);
        }
        sessionRepository.setDefaultMaxInactiveInterval(this.maxInactiveIntervalInSeconds);
        sessionRepository.setFlushMode(this.flushMode);
        sessionRepository.setSaveMode(this.saveMode);
        if (this.indexResolver != null) {
            sessionRepository.setIndexResolver(this.indexResolver);
        }
        if (this.lobHandler != null) {
            sessionRepository.setLobHandler(this.lobHandler);
        } else if (requiresTemporaryLob(this.dataSource)) {
            DefaultLobHandler defaultLobHandler = new DefaultLobHandler();
            defaultLobHandler.setCreateTemporaryLob(true);
            sessionRepository.setLobHandler(defaultLobHandler);
        }
        if (this.springSessionConversionService != null) {
            sessionRepository.setConversionService(this.springSessionConversionService);
        } else if (this.conversionService != null) {
            sessionRepository.setConversionService(this.conversionService);
        } else {
            sessionRepository.setConversionService(createConversionServiceWithBeanClassLoader(this.classLoader));
        }
        this.sessionRepositoryCustomizers
                .forEach(sessionRepositoryCustomizer -> sessionRepositoryCustomizer.customize(sessionRepository));
        return sessionRepository;
    }

    @Bean
    public SessionRegistry sessionRegistry() {
        return new SpringSessionBackedSessionRegistry<>(this.sessionRepository());
    }

    @SuppressWarnings("deprecation")
    private static boolean requiresTemporaryLob(DataSource dataSource) {
        try {
            String productName = JdbcUtils.extractDatabaseMetaData(dataSource, "getDatabaseProductName");
            return "Oracle".equalsIgnoreCase(JdbcUtils.commonDatabaseName(productName));
        } catch (MetaDataAccessException ex) {
            return false;
        }
    }

    public void setMaxInactiveIntervalInSeconds(Integer maxInactiveIntervalInSeconds) {
        this.maxInactiveIntervalInSeconds = maxInactiveIntervalInSeconds;
    }

    public void setTableName(String tableName) {
        this.tableName = tableName;
    }

    public void setCleanupCron(String cleanupCron) {
        this.cleanupCron = cleanupCron;
    }

    public void setFlushMode(FlushMode flushMode) {
        this.flushMode = flushMode;
    }

    public void setSaveMode(SaveMode saveMode) {
        this.saveMode = saveMode;
    }

    @Autowired
    public void setDataSource(@SpringSessionDataSource ObjectProvider<DataSource> springSessionDataSource,
                              ObjectProvider<DataSource> dataSource) {
        DataSource dataSourceToUse = springSessionDataSource.getIfAvailable();
        if (dataSourceToUse == null) {
            dataSourceToUse = dataSource.getObject();
        }
        this.dataSource = dataSourceToUse;
    }

    @Autowired
    public void setTransactionManager(PlatformTransactionManager transactionManager) {
        this.transactionManager = transactionManager;
    }

    @Autowired(required = false)
    @Qualifier("springSessionTransactionOperations")
    public void setTransactionOperations(TransactionOperations transactionOperations) {
        this.transactionOperations = transactionOperations;
    }

    @Autowired(required = false)
    public void setIndexResolver(IndexResolver<Session> indexResolver) {
        this.indexResolver = indexResolver;
    }

    @Autowired(required = false)
    @Qualifier("springSessionLobHandler")
    public void setLobHandler(LobHandler lobHandler) {
        this.lobHandler = lobHandler;
    }

    @Autowired(required = false)
    @Qualifier("springSessionConversionService")
    public void setSpringSessionConversionService(ConversionService conversionService) {
        this.springSessionConversionService = conversionService;
    }

    @Autowired(required = false)
    @Qualifier("conversionService")
    public void setConversionService(ConversionService conversionService) {
        this.conversionService = conversionService;
    }

    @Autowired(required = false)
    public void setSessionRepositoryCustomizer(
            ObjectProvider<SessionRepositoryCustomizer<DeployJdbcIndexedSessionRepository>> sessionRepositoryCustomizers) {
        this.sessionRepositoryCustomizers = sessionRepositoryCustomizers.orderedStream().collect(Collectors.toList());
    }

    @Override
    public void setBeanClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;
    }

    @Autowired(required = false)
    public void setSpringSessionConverterFactory(SpringSessionConverterFactory springSessionConverterFactory) {
        this.springSessionConverterFactory = springSessionConverterFactory;
    }

    @Override
    public void setEmbeddedValueResolver(StringValueResolver resolver) {
        this.embeddedValueResolver = resolver;
    }

    @Override
    public void setImportMetadata(AnnotationMetadata importMetadata) {
        Map<String, Object> attributeMap = importMetadata
                .getAnnotationAttributes(EnableJdbcHttpSession.class.getName());
        if (attributeMap != null) {
            AnnotationAttributes attributes = AnnotationAttributes.fromMap(attributeMap);
            this.maxInactiveIntervalInSeconds = attributes.getNumber("maxInactiveIntervalInSeconds");
            String tableNameValue = attributes.getString("tableName");
            if (StringUtils.hasText(tableNameValue)) {
                this.tableName = this.embeddedValueResolver.resolveStringValue(tableNameValue);
            }
            this.flushMode = attributes.getEnum("flushMode");
            this.saveMode = attributes.getEnum("saveMode");
        }
    }

    private static JdbcTemplate createJdbcTemplate(DataSource dataSource) {
        DeployJdbcTemplate jdbcTemplate = new DeployJdbcTemplate(dataSource, true);
        jdbcTemplate.afterPropertiesSet();
        return jdbcTemplate;
    }

    private static TransactionTemplate createTransactionTemplate(PlatformTransactionManager transactionManager) {
        TransactionTemplate transactionTemplate = new TransactionTemplate(transactionManager);
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionTemplate.afterPropertiesSet();
        return transactionTemplate;
    }

    private GenericConversionService createConversionServiceWithBeanClassLoader(ClassLoader classLoader) {
        GenericConversionService conversionService = new GenericConversionService();
        if (this.springSessionConverterFactory != null) {
            conversionService.addConverter(this.springSessionConverterFactory.buildSerializer(classLoader));
            conversionService.addConverter(this.springSessionConverterFactory.buildDeserializer(classLoader));
        } else {
            conversionService.addConverter(Object.class, byte[].class, new SerializingConverter());
            conversionService.addConverter(byte[].class, Object.class, new DeserializingConverter(classLoader));
        }
        return conversionService;
    }

    @Configuration(proxyBeanMethods = false)
    @Primary
    static class SpringBootJdbcHttpJdbcSessionConfiguration extends SpringJdbcSessionConfig {

        @Autowired
        void customize(SessionProperties sessionProperties, JdbcSessionProperties jdbcSessionProperties,
                       ServerProperties serverProperties) {
            Duration timeout = sessionProperties
                    .determineTimeout(() -> serverProperties.getServlet().getSession().getTimeout());
            if (timeout != null) {
                setMaxInactiveIntervalInSeconds((int) timeout.getSeconds());
            }
            setTableName(jdbcSessionProperties.getTableName());
            setCleanupCron(jdbcSessionProperties.getCleanupCron());
            setFlushMode(jdbcSessionProperties.getFlushMode());
            setSaveMode(jdbcSessionProperties.getSaveMode());
        }
    }

    /**
     * Configuration of scheduled job for cleaning up expired sessions.
     */
    @EnableScheduling
    @Configuration(proxyBeanMethods = false)
    class SessionCleanupConfiguration implements SchedulingConfigurer {

        private final DeployJdbcIndexedSessionRepository sessionRepository;

        SessionCleanupConfiguration(DeployJdbcIndexedSessionRepository sessionRepository) {
            this.sessionRepository = sessionRepository;
        }

        @Override
        public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
            taskRegistrar.addCronTask(this.sessionRepository::cleanUpExpiredSessions,
                    SpringJdbcSessionConfig.this.cleanupCron);
        }

    }
}
