package com.xebialabs.deployit.core.ordering

import org.springframework.core.annotation.Order
import org.springframework.stereotype.Component

import scala.annotation.tailrec
import scala.util.matching.Regex
import scala.util.{Failure, Success, Try}

trait PartialComparator[T] {
  def compare(left: T, right: T): ComparisonResult

  trait ComparisonResult {
    def processedLeft: Boolean

    def processedRight: Boolean

    def processResult: Int
  }

}

abstract class ListBasedPartialComparator[T] extends PartialComparator[T] {

  override def compare(left: T, right: T): ComparisonResult = {
    val resLeft = calculateSeq(left)
    val resRight = calculateSeq(right)
    new ComparisonResult {
      override def processedLeft: Boolean = resLeft.nonEmpty

      override def processedRight: Boolean = resRight.nonEmpty

      override def processResult: Int = doCompare(resLeft, resRight)
    }
  }

  def doCompare(l1: Seq[Either[String, Int]], l2: Seq[Either[String, Int]]): Int = {
    l1.zip(l2).map {
      case (Left(null), Left(null)) => 0
      case (Left(null), Left(_)) => 1
      case (Left(null), _) => -1
      case (Left(_), Left(null)) => -1
      case (_, Left(null)) => 1
      case (Left(s1), Left(s2)) => s1.compareTo(s2)
      case (Right(i1), Right(i2)) => i1.compareTo(i2)
      case (Left(_), Right(_)) => -1
      case (Right(_), Left(_)) => 1
    }.collectFirst { case n if n != 0 => n }.getOrElse(0)
  }

  def calculateSeq(t: T): Seq[Either[String, Int]]
}

class RegexComparator(r: Regex) extends ListBasedPartialComparator[String] {
  override def calculateSeq(t: String): Seq[Either[String, Int]] =
    r.findAllMatchIn(t).flatMap { m => m.groupNames.zipAll(m.subgroups, null, null) }.map {
      case (null, group) => intOrString(group)
      case (groupName, group) if groupName.startsWith("int") => intOrString(group)
      case (groupName, group) if groupName.startsWith("string") => string(group)
    }.toSeq

  private[this] def intOrString(group: String): Either[String, Int] =
    Try(Integer.parseInt(group)) match {
      case Success(value) => Right(value)
      case Failure(_: NumberFormatException) => Left(group)
      case Failure(ex) => throw ex
    }

  private[this] def string(group: String): Either[String, Int] = Left(group)
}

@Order(0)
@Component
class SemVerComparator extends {
  private val NUM = "(0|[1-9][0-9]*)"
} with RegexComparator(new Regex(s"^$NUM(?:\\.$NUM(?:\\.$NUM)?)?(?:-([0-9a-zA-Z-]+)?(?:\\.$NUM)?)?$$",
  "int-major", "int-minor", "int-patch", "string-pre-release", "int-pre-release-order"))

@Order(2)
@Component
class LexicographicalComparator extends PartialComparator[String] {
  override def compare(left: String, right: String): ComparisonResult = new ComparisonResult {
    override def processedRight: Boolean = true
    override def processedLeft: Boolean = true
    override def processResult: Int = left compareTo right
  }
}

@Order(1)
@Component
class NumericComparator extends ListBasedPartialComparator[String] {

  override def calculateSeq(t: String): Seq[Either[String, Int]] = Try {
    t.split("\\.").toList.map(e => Right(Integer.parseInt(e)))
  } match {
    case Success(result: Seq[Either[String, Int]]) => result
    case Failure(_: NumberFormatException) => List.empty[Either[String,Int]]
    case Failure(ex) => throw ex

  }
}
