package com.xebialabs.deployit.core.config

import java.sql.Connection
import java.util

import com.tqdev.metrics.core.MetricRegistry
import com.tqdev.metrics.jdbc.InstrumentedDataSource
import com.xebialabs.deployit.core.config.SqlConfiguration.hikariDataSource
import com.xebialabs.deployit.core.config.db.{DatabaseProperties, MainDatabase, ReportingProperties}
import com.xebialabs.deployit.core.metrics.XldDbMetricsTrackerFactory
import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.config.DatabaseDriverConfiguration
import com.xebialabs.deployit.core.sql.spring.DeployJdbcTemplate
import com.zaxxer.hikari.{HikariConfig, HikariDataSource}
import grizzled.slf4j.Logging
import javax.annotation.PostConstruct
import javax.sql.DataSource
import liquibase.Liquibase
import liquibase.database.jvm.JdbcConnection
import liquibase.ext.ClassLoaderForLiquibase
import org.springframework.beans.factory.annotation.{Autowired, Qualifier, Value}
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.context.annotation.{Bean, Configuration, Import, Primary}
import org.springframework.dao.annotation.PersistenceExceptionTranslationPostProcessor
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.datasource.DataSourceTransactionManager
import org.springframework.orm.jpa.vendor.HibernateJpaVendorAdapter
import org.springframework.orm.jpa.{AbstractEntityManagerFactoryBean, JpaTransactionManager, LocalContainerEntityManagerFactoryBean}
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.EnableTransactionManagement


@Configuration
@EnableTransactionManagement
@EnableConfigurationProperties
@Import(Array(
  classOf[MainDatabase],
  classOf[ReportingProperties],
  classOf[XldDbMetricsTrackerFactory]
))
class SqlConfiguration extends Logging {

  @Autowired(required = true)
  var mainDatabase: MainDatabase = _

  @Autowired(required = true)
  var reportingProperties: ReportingProperties = _

  @Autowired
  var xldDbMetricsTrackerFactory: XldDbMetricsTrackerFactory = _

  @Value("${deploy.internal.db.changelog.master.filename:db/changelog/db.changelog-master.yaml}")
  var changeLogMasterFilename: String = _

  @Value("${deploy.internal.db.changelog.reporting.filename:db/changelog/db.changelog-reporting.yaml}")
  var changeLogReportingFilename: String = _

  @Primary
  @Bean(destroyMethod = "close")
  def mainDataSource: DataSource = {
    DatabaseDriverConfiguration.mainDatabaseDriverClassName = mainDatabase.database.dbDriverClassname
    instrumentedDataSource(hikariDataSource(mainDatabase.database, "MainPool", xldDbMetricsTrackerFactory))
  }

  @Bean(destroyMethod = "close")
  def reportingDataSource: DataSource = {
    val database = if (reportingProperties.database.hasConfigured) reportingProperties.database else mainDatabase.database
    DatabaseDriverConfiguration.reportingDatabaseDriverClassName = database.dbDriverClassname
    instrumentedDataSource(hikariDataSource(database, "ReportingPool", xldDbMetricsTrackerFactory))
  }

  private def instrumentedDataSource(dataSource: DataSource) = new InstrumentedDataSource(dataSource, MetricRegistry.getInstance())

  @Bean
  def mainSchema: SchemaInfo = {
    val dialect = SqlDialect.initializeDialect(mainDataSource)
    logger.info(s"Detected SQL dialect for main: $dialect.")
    SchemaInfo(dialect, Option(mainDatabase.database.dbSchemaName).orElse(None))
  }

  @Bean
  def reportingSchema: SchemaInfo = {
    val dialect = SqlDialect.initializeDialect(reportingDataSource)
    logger.info(s"Detected SQL dialect for reporting: $dialect.")
    SchemaInfo(dialect, Option(reportingProperties.database.dbSchemaName).orElse(None))
  }

  @Bean
  @Primary
  def mainJdbcTemplate: JdbcTemplate = new DeployJdbcTemplate(mainDataSource, true)

  @Bean
  def entityManagerFactory(): AbstractEntityManagerFactoryBean = {
    val em = new LocalContainerEntityManagerFactoryBean
    em.setDataSource(mainDataSource)
    em.setJpaPropertyMap(getJpaProperties(mainDatabase.database.dbDriverClassname, em))
    em.setPackagesToScan("com.xebialabs.deployit")
    val vendorAdapter = new HibernateJpaVendorAdapter
    em.setJpaVendorAdapter(vendorAdapter)
    em
  }

  def getJpaProperties(driverClassName: String, em: LocalContainerEntityManagerFactoryBean): util.Map[String, AnyRef] = {
    val properties: util.Map[String, AnyRef] = em.getJpaPropertyMap
    if (driverClassName.contains(JpaUtils.DERBY_JDBC_DRIVER_CLASS_PATTERN)) {
      properties.put(JpaUtils.HIBERNATE_DIALECT_KEY, JpaUtils.DERBY_DIALECT_CLASS)
    } else if (driverClassName.contains(JpaUtils.DB2_JDBC_DRIVER_CLASS_PATTER)) {
      properties.put(JpaUtils.HIBERNATE_DIALECT_KEY, JpaUtils.DB2_DIALECT_CLASS)
    }
    properties
  }

  @Bean
  def exceptionTranslation = new PersistenceExceptionTranslationPostProcessor

  @Primary
  @Bean
  def mainTransactionManager: PlatformTransactionManager = new JpaTransactionManager(entityManagerFactory().getObject)

  @Bean
  def reportingTransactionManager: PlatformTransactionManager =
    new DataSourceTransactionManager(reportingDataSource)

  @PostConstruct
  def initializeMainDatabase(): Unit =
    updateLiquibase(changeLogMasterFilename, mainDataSource, mainSchema.sqlDialect)

  @PostConstruct
  def initializeReportingDatabase(): Unit =
    updateLiquibase(changeLogReportingFilename, reportingDataSource, reportingSchema.sqlDialect)

  private def updateLiquibase(schema: String, dataSource: DataSource, dialect: SqlDialect): Unit = {
    import SqlConfiguration.liquibaseContext
    val connection = dataSource.getConnection
    try {
      val context = liquibaseContext(dialect, connection)
      logger.info(s"Detected liquibase context: $context")
      new Liquibase(
        schema,
        new ClassLoaderForLiquibase(),
        new JdbcConnection(connection)
      ).update(context)
    } finally {
      connection.close()
    }
  }
}

object SqlConfiguration {

  def hikariDataSource(config: DatabaseProperties, name: String, xldDbMetricsTrackerFactory: XldDbMetricsTrackerFactory) =
    new HikariDataSource(hikariConfig(config, name, xldDbMetricsTrackerFactory))

  private def hikariConfig(config: DatabaseProperties, poolName: String, xldDbMetricsTrackerFactory: XldDbMetricsTrackerFactory): HikariConfig = {
    val hikariConfig = new HikariConfig()
    var url = config.dbUrl
    // Microsoft SQL DB specific property sendStringParametersAsUnicode
    if (DatabaseDriverConfiguration.isMSSqlDatabase(config.dbDriverClassname)
      && !url.contains("sendStringParametersAsUnicode")) {
      url = url.concat(";sendStringParametersAsUnicode=false")
    }
    hikariConfig.setDriverClassName(config.dbDriverClassname)
    hikariConfig.setJdbcUrl(url)
    hikariConfig.setUsername(config.dbUsername)
    hikariConfig.setPassword(config.dbPassword)
    hikariConfig.setMaximumPoolSize(config.maxPoolSize)
    hikariConfig.setMaxLifetime(config.getMaxLifeTimeInMillis)
    hikariConfig.setIdleTimeout(config.getIdleTimeoutInMillis)
    hikariConfig.setMinimumIdle(config.minimumIdle)
    hikariConfig.setConnectionTimeout(config.getConnectionTimeoutInMillis)
    hikariConfig.setTransactionIsolation("TRANSACTION_READ_COMMITTED")
    hikariConfig.setPoolName(poolName)
    hikariConfig.setAutoCommit(false)
    hikariConfig.setLeakDetectionThreshold(config.getLeakDetectionThresholdInMillis)
    hikariConfig.setMetricsTrackerFactory(xldDbMetricsTrackerFactory)
    hikariConfig
  }

  def liquibaseContext(dialect: SqlDialect, con: Connection): String = {
    val majorVersion = con.getMetaData.getDatabaseMajorVersion
    s"${dialect.toString}-$majorVersion"
  }
}

object XldMetricRegistry {
  val metricRegistry = new MetricRegistry()
}

object JpaUtils {
  val HIBERNATE_DIALECT_KEY = "hibernate.dialect"
  val DERBY_DIALECT_CLASS = "com.xebialabs.deployit.hibernate.dialect.DeployDerbyDialect"
  val DB2_DIALECT_CLASS = "org.hibernate.dialect.DB297Dialect"
  val DERBY_JDBC_DRIVER_CLASS_PATTERN = "org.apache.derby"
  val DB2_JDBC_DRIVER_CLASS_PATTER = "com.ibm.db2"
}
