package com.xebialabs.deployit.core.config

import java.util.{List => JList}
import com.fasterxml.jackson.annotation.JsonInclude.Include
import com.fasterxml.jackson.annotation.{JsonInclude, JsonProperty}
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.stereotype.Component

import scala.beans.BeanProperty

/**
  * Server specific configuration for a single deploy server master instance.
  */
@Component
@ConfigurationProperties(prefix = "deploy.server")
@JsonInclude(Include.NON_NULL)
class ServerSideProperties {

  /**
    * The resolvable DNS name or IP address.
    */
  @BeanProperty
  @JsonProperty("hostname")
  var hostName: String = "127.0.0.1"

  /**
    * The host name to listen on.
    * The default setting will listen on all network interfaces.
    * A specific interface can be targeted by name or address.
    */
  @BeanProperty
  @JsonProperty("bind-hostname")
  var bindHostName: String = "0.0.0.0"

  /**
    * The port for communication with the worker nodes
    */
  @BeanProperty
  @JsonProperty
  var port: Int = 8180

  /**
    * The label of the master instance.
    */
  @BeanProperty
  @JsonProperty
  var label: String = ""

  /**
    * Configuration of sending events
    */
  @BeanProperty
  @JsonProperty
  var events = new Events

  @BeanProperty
  @JsonProperty
  var ssl = new SslConfiguration

  @BeanProperty
  @JsonProperty
  var session = new ServerSideSession

  @BeanProperty
  @JsonProperty("aggregation-timeout")
  var aggregationTimeout = "5 seconds"

  @BeanProperty
  @JsonProperty
  var license = new License

  /**
    * Where to cache download results for exported DARs and reports
    */
  @BeanProperty
  @JsonProperty
  var downloads = new Downloads

  @BeanProperty
  @JsonProperty
  var algorithms = new Algorithms

  @BeanProperty
  @JsonProperty("patch-dictionaries")
  var patchDictionaries = new PatchDictionaries

  /**
    * Enables the server to resolve application dependencies.
    *
    * Default: true
    */
  @BeanProperty
  @JsonProperty("resolve-application-dependencies-enabled")
  var resolveApplicationDependenciesEnabled: Boolean = true

}

class Downloads {
  /**
    * The absolute folder name or relative to installation folder
    */
  @BeanProperty
  @JsonProperty("export-root")
  var exportRoot: String = "export"
}

class License {
  /**
    * License expiry warning (in days).
    *
    * Default: 5
    */
  @BeanProperty
  @JsonProperty
  var daysBeforeWarning: Int = 5
}

class ServerSideSession {
  /**
   * Configures to clean up expired user sessions with the specified cron.
   *
   * Default: "0 0/20 * * * *"
   */
  @BeanProperty
  @JsonProperty("cleanup-cron")
  var cleanupCron: String = "0 0/20 * * * *"

  /**
   * To disable the “Active Sessions” screen, set this property to false. Once turned off the feature also stops all associated data collection.
   * If you are using MSSQL, we recommend that you disable “Active Sessions” to prevent deadlocks in the tables.
   *
   * Default: true
   */
  @BeanProperty
  @JsonProperty("active-user-sessions-enabled")
  var activeUserSessionsEnabled: Boolean = true
}

@JsonInclude(Include.NON_EMPTY)
class SslConfiguration {

  /**
    * Should be set to true to enable SSL.
    *
    * Default: false
    */
  @BeanProperty
  var enabled: Boolean = false

  /**
    * Example: ["TLS_RSA_WITH_AES_128_CBC_SHA", "TLS_RSA_WITH_AES_256_CBC_SHA"]
    * You need to install the JCE Unlimited Strength Jurisdiction Policy
    * Files to use AES 256.
    * More info here:
    * http://docs.oracle.com/javase/7/docs/technotes/guides/security/SunProviders.html#SunJCEP
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("enabled-algorithms")
  var enabledAlgorithms: JList[String] = JList.of()

  /**
    * Path to the key store.
    *
    * Default: keystore
    */
  @BeanProperty
  @JsonProperty("key-store")
  var keyStore: String = "keystore"

  /**
    * Password for the key.
    *
    * Default: changeme
    */
  @BeanProperty
  @JsonProperty("key-password")
  var keyPassword: String = "changeme"

  /**
    * Password for the key store.
    *
    * Default: changeme
    */
  @BeanProperty
  @JsonProperty("key-store-password")
  var keyStorePassword: String = "changeme"

  /**
    * Is mutual SSL enabled.
    *
    * Default: false
    */
  @BeanProperty
  @JsonProperty("mutual-enabled")
  var mutualEnabled: Boolean = false

  /**
    * Protocol to use for SSL encryption.
    *
    * For example: 'TLSv1.1', 'TLSv1.2'
    *
    * Default: "TLSv1.2"
    *
    */
  @BeanProperty
  @JsonProperty
  var protocol: String = "TLSv1.2"

  /**
    * (Docs taken from https://doc.akka.io/docs/akka/2.6/general/configuration-reference.html)
    * There are two options, and the default SecureRandom is recommended:
    * "" or "SecureRandom" => (default)
    * "SHA1PRNG" => Can be slow because of blocking issues on Linux
    *
    * "AES128CounterSecureRNG" and "AES256CounterSecureRNG" were deprecated in Akka
    * 2.5.16 and if these configuration values are used it will use the default
    * SecureRandom anyway and log a warning. The deprecated implementations are
    * not recommended, but they can be enabled by using configuration values
    * "DeprecatedAES128CounterSecureRNG" or "DeprecatedAES256CounterSecureRNG"
    *
    * Setting a value here may require you to supply the appropriate cipher
    * suite (see enabled-algorithms section above)
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("random-number-generator")
  var randomNumberGenerator: String = ""

  /**
    * Path to a trust store.
    *
    * Default: truststore
    */
  @BeanProperty
  @JsonProperty("trust-store")
  var trustStore: String = "truststore"

  /**
    * Password for a key store.
    *
    * Default: changeme
    */
  @BeanProperty
  @JsonProperty("trust-store-password")
  var trustStorePassword: String = "changeme"
}

class Events {

  /**
    * The time to live of events in the queue before it's going to be pruned.
    * As after that period of time the event can be outdated for the system and giving a not relevant information.
    *
    * Default: 5 minutes
    */
  @BeanProperty
  @JsonProperty("time-to-live")
  var timeToLive: String = "5 minutes"

  /**
    * The name of the queue where events related to CIs changes are sent.
    *
    * Default: xld-cis-changed
    */
  @BeanProperty
  @JsonProperty("cis-changed-queue-name")
  var cisChangedQueueName: String = "xld-cis-changed"

}

class Algorithms {
  /**
    * The algorithm for checksum computation, can be SHA-256 or SHA-1 (legacy)
    * SHA-256 is a default value starting from 9.5.0 release
    * SHA-256 should always be used for any new installation for security reasons
    * SHA-256 cannot be used for installations with existing file/folder/archive or any other artifacts with computed checksum from previous releases, use SHA-1 in this case
    *
    * The migration procedure will determine if SHA-256 can be used and will set SHA-1 otherwise
    *
    * Default: SHA-256
    */
  @BeanProperty
  @JsonProperty
  var checksum: String = "SHA-256"
}

class PatchDictionaries {
  /**
    * The limitation for patch-dictionary file which can be uploaded (in Mb).
    *
    * Default: 5
    */
  @BeanProperty
  @JsonProperty("max-file-size-mb")
  var maxFileSize: Int = 5
}

object ServerSideProperties {

  var DEFAULT_WEB_CONTEXT_ROOT: String = "/"

  var DEFAULT_HTTP_PORT: Int = 4516

  var DEFAULT_HTTP_BIND_ADDRESS: String = "0.0.0.0"

  var DEFAULT_MAX_THREADS: Int = 150

  var DEFAULT_MIN_THREADS: Int = 30

  var KEY_HTTP_BIND_ADDRESS: String = "http.bind.address"

  var KEY_HTTP_PORT: String = "http.port"

  var KEY_HTTP_CONTEXT_ROOT: String = "http.context.root"

  var KEY_IMPORTABLE_PACKAGES_PATH: String = "importable.packages.path"

  var KEY_MINIMUM_THREADS: String = "threads.min"

  var KEY_MAXIMUM_THREADS: String = "threads.max"

  var KEY_ACTIVE_USER_SESSIONS: String = "active.user.sessions"
}
