package com.xebialabs.deployit.core.config

import java.beans.BeanProperty
import java.util.{List => JList}

import com.fasterxml.jackson.annotation.JsonProperty
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

/**
  * Configuration to monitor the system via Kamon.
  */
@Configuration
@ConfigurationProperties(prefix = "deploy.jmx")
class JmxProperties {

  /**
    * Enabling the JMX.
    *
    * Default: no
    */
  @BeanProperty
  @JsonProperty("enabled")
  var enabled: String = "no"

  /**
    * Host name for RMI server
    *
    * No default value
    */
  @BeanProperty
  @JsonProperty("hostname")
  var hostname: String = _

  /**
    * Port of the RMI server
    *
    * Default: 1099
    */
  @BeanProperty
  @JsonProperty("port")
  var port: Int = 1099

  @BeanProperty
  @JsonProperty("ssl")
  var ssl: SSL = _

  /**
    * Internal configuration of Kamon.
    *
    * Check the tool itself for more information: https://github.com/kamon-io/kamon-jmx
    */
  @BeanProperty
  @JsonProperty("kamon")
  var kamon: Kamon = _

}

class SSL {
  /**
    * Should be set to true to enable SSL.
    *
    * Default: false
    */
  @BeanProperty
  @JsonProperty("enabled")
  var enabled: String = "false"

  /**
    * Path to the key store.
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("key-store")
  var keyStore: String = _

  /**
    * Password for the key store.
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("key-store-password")
  var keyStorePassword: String = _

  /**
    * Path to a trust store.
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("trust-store")
  var trustStore: String = _

  /**
    * Password for a key store.
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("trust-store-password")
  var trustStorePassword: String = _

  /**
    * Protocol to use for SSL encryption.
    *
    * For example: 'TLSv1.1', 'TLSv1.2'.
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("protocol")
  var protocol: String = _

  /**
    * Example: ["TLS_RSA_WITH_AES_128_CBC_SHA", "TLS_RSA_WITH_AES_256_CBC_SHA"]
    * You need to install the JCE Unlimited Strength Jurisdiction Policy
    * Files to use AES 256.
    * More info here:
    * http://docs.oracle.com/javase/7/docs/technotes/guides/security/SunProviders.html#SunJCEP
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("enabled-algorithms")
  var enabledAlgorithms: JList[String] = JList.of()

  /**
    * (Docs taken from https://doc.akka.io/docs/akka/2.6/general/configuration-reference.html)
    * There are two options, and the default SecureRandom is recommended:
    * "" or "SecureRandom" => (default)
    * "SHA1PRNG" => Can be slow because of blocking issues on Linux
    *
    * "AES128CounterSecureRNG" and "AES256CounterSecureRNG" were deprecated in Akka
    * 2.5.16 and if these configuration values are used it will use the default
    * SecureRandom anyway and log a warning. The deprecated implementations are
    * not recommended, but they can be enabled by using configuration values
    * "DeprecatedAES128CounterSecureRNG" or "DeprecatedAES256CounterSecureRNG"
    *
    * Setting a value here may require you to supply the appropriate cipher
    * suite (see enabled-algorithms section above)
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("random-number-generator")
  var randomNumberGenerator: String = _
}

class Kamon {

  @BeanProperty
  @JsonProperty("show-aspectj-missing-warning")
  var showAspectjMissingWarning: String = "yes"

  @BeanProperty
  @JsonProperty("metric")
  var metric: Metric = _

  @BeanProperty
  @JsonProperty("kamon-mxbeans")
  var kamonMxbeans: KamonMxbeans = _

}

class Metric {

  @BeanProperty
  @JsonProperty("track-unmatched-entities")
  var trackUnmatchedEntities: String = _

  @BeanProperty
  @JsonProperty("filters")
  var filters: Filters = _
}

class Filters {

  /**
    * {{{
    * includes:
    *    - "PlannerActorSystem/\*\*"
    *    - "task-sys/\*\*"
    *    - "xl-scheduler-system/\*\*"
    * excludes: []
    * }}}
    */
  @BeanProperty
  @JsonProperty("akka-dispatcher")
  var akkaDispatcher: AkkaDispatcher = _
}

class AkkaDispatcher {

  /**
    * Default: empty
    */
  @BeanProperty
  @JsonProperty("includes")
  var includes: JList[String] = JList.of()

  /**
    * Default: empty
    */
  @BeanProperty
  @JsonProperty("excludes")
  var excludes: JList[String] = JList.of()
}

class KamonMxbeans {

  @BeanProperty
  @JsonProperty("mbeans")
  var mbeans: JList[String] = JList.of()

  @BeanProperty
  @JsonProperty("identify-delay-interval-ms")
  var identifyDelayIntervalMs: Int = 1000

  @BeanProperty
  @JsonProperty("identify-interval-ms")
  var identifyIntervalMs: Int = 1000

  @BeanProperty
  @JsonProperty("value-check-interval-ms")
  var valueCheckIntervalMs: Int = 1000
}
