package com.xebialabs.deployit.core.config

import java.util.{List => JList}

import com.fasterxml.jackson.annotation.JsonProperty
import com.xebialabs.deployit.core.config.ClientProperties._
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

/**
  * This configuration section contains the configuration which can adjust the behavior of the application in certain
  * areas.
  */
@Configuration
@ConfigurationProperties(prefix = "deploy.client")
class ClientProperties {

  /**
    * When set to “true”, all deployables will be auto-mapped to containers when you set up an initial or
    * update deployment in the GUI, and Deploy will ignore
    * the map.all.deployables.on.initial and map.all.deployables.on.update settings.
    *
    * Default: true
    */
  @BeanProperty
  @JsonProperty("automatically-map-all-deployables")
  var automaticallyMapAllDeployables: Boolean = true

  /**
    * When set to “true”, all deployables will be auto-mapped to containers
    * only when you set up an initial deployment in the GUI.
    *
    * Default: false
    */
  @BeanProperty
  @JsonProperty("automatically-map-all-deployables-on-initial")
  var automaticallyMapAllDeployablesOnInitial: Boolean = false

  /**
    * When set to “true”, all deployables will be auto-mapped to containers only when you set up an update deployment.
    *
    * Default: false
    */
  @BeanProperty
  @JsonProperty("automatically-map-all-deployables-on-update")
  var automaticallyMapAllDeployablesOnUpdate: Boolean = false

  @BeanProperty
  var gui: ClientGuiConfiguration = new ClientGuiConfiguration

  /**
    * The location where importable packages will be searched in.
    * One of the way how the package can be imported is, by executing in CLI: deployit.importPackage,
    * there is also an option in GUI on `Application` node.
    *
    * Default: importablePackages
    */
  @BeanProperty
  @JsonProperty("importable-package-directory")
  var importablePackageDirectory: String = DEFAULT_IMPORTABLE_PACKAGES_PATH

  @BeanProperty
  var maintenance: Maintenance = new Maintenance()

  /**
    * Sets if a Cookie marked as secure should be used.
    *
    * Default: false
    */
  @BeanProperty
  @JsonProperty("secure-cookie-enabled")
  var secureCookieEnabled: Boolean = false

  /**
    * This option toggle on and off the possibility for users to use "Remember me" feature.
    * In case of toggled off, the option even won't be visible on GUI.
    *
    * Default: true
    */
  @BeanProperty
  @JsonProperty("session-remember-enabled")
  var sessionRememberEnabled: Boolean = true

  @BeanProperty
  @JsonProperty
  var session = new ClientSession

  /**
    * This option allows to tune the bulk update for upgraders.
    * It means how many Configuration Items will be updated in a single loop.
    * Mind of database dialect limitations for IN CLAUSE before configuring too high value.
    *
    * Default: 100
    */
  @BeanProperty
  @JsonProperty("upgrade-batch-size")
  var upgradeBatchSize: Int = 100

  /**
    * Uploaded files buffer.
    * It is used for example whenever CI is created which contains the artifact.
    *
    * Default: 262144
    */
  @BeanProperty
  @JsonProperty("upload-file-buffer-size")
  var uploadFileBufferSize: Int = DEFAULT_UPLOAD_FILE_BUFFER_SIZE

  /**
    * Hide or show any internal server errors due to incorrect user input.
    *
    * Default: false
    */
  @BeanProperty
  @JsonProperty("hide-internals")
  var hideInternals: Boolean = false

}

class Maintenance {

  /**
    * During the maintenance certain limitations are applied.
    * This configuration allows to tailor the list of forbidden API paths for users during maintenance mode.
    * Admin user is not affected.
    *
    * Default: deployit/task:POST,deployit/tasks/v2:POST,deployit/control:POST,xldeploy/task:POST,xldeploy/tasks/v2:POST,xldeploy/control:POST
    */
  @BeanProperty
  @JsonProperty("forbidden-paths")
  var forbiddenPaths: JList[String] = DEFAULT_MAINTENANCE_FORBIDDEN_REQUESTS.split(",").toList.asJava
}

class ClientGuiConfiguration {

  @BeanProperty
  var toastr: Toastr = new Toastr

  /**
    * Which landing page to display as initial.
    *
    * Two options are available: "explorer" and "deploymentWorkspace".
    *
    * Default: explorer
    */
  @BeanProperty
  @JsonProperty("landing-page")
  var landingPage: String = "explorer"

  @BeanProperty
  var login: Login = new Login()

  // TODO: currently not used, add JavaDoc with https://digitalai.atlassian.net/browse/ENG-6455
  @BeanProperty
  var task = new {
    @BeanProperty
    var status = new {
      @BeanProperty
      var poll = new {
        @BeanProperty
        @JsonProperty("timeout-ms")
        var timeoutMs: Int = 1000
      }
    }
  }
}

class ClientSession {
  /**
    * Number of minutes before a user’s session is locked when the GUI is idle. If the value is set to 0, and a user session is inactive for more than 30 days,
    * it will be automatically purged from the session database.
    *
    * Default: 0 (no timeout)
    */
  @BeanProperty
  @JsonProperty("timeout-minute")
  var timeoutMinute: Int = 20
}

class Toastr {

  /**
    * Configuration for error toastr.
    */
  @BeanProperty
  @JsonProperty("error")
  var error = new {
    /**
      * How long (in ms) to display an error toastr to the user before to auto-hide it.
      * 0 is to turn off auto-hiding
      *
      * Default: 0
      */
    @BeanProperty
    @JsonProperty("timeout-ms")
    var timeoutMs: Int = 0
  }

  /**
    * Configuration for info toastr.
    */
  @BeanProperty
  @JsonProperty("info")
  var info = new {
    /**
      * How long (in ms) to display an information toastr to the user before to auto-hide it.
      * 0 is to turn off auto-hiding
      *
      * Default: 10000
      */
    @BeanProperty
    @JsonProperty("timeout-ms")
    var timeoutMs: Int = 10000
  }

  /**
    * Configuration for success toastr.
    */
  @BeanProperty
  @JsonProperty("success")
  var success = new {
    /**
      * How long (in ms) to display an successful toastr to the user before to auto-hide it.
      * 0 is to turn off auto-hiding
      *
      * Default: 10000
      */
    @BeanProperty
    @JsonProperty("timeout-ms")
    var timeoutMs: Int = 10000
  }

  /**
    * Configuration for warning toastr.
    */
  @BeanProperty
  @JsonProperty("warning")
  var warning = new {
    /**
      * How long (in ms) to display an warning toastr to the user before to auto-hide it.
      * 0 is to turn off auto-hiding
      *
      * Default: 10000
      */
    @BeanProperty
    @JsonProperty("timeout-ms")
    var timeoutMs: Int = 10000
  }

}

class Login {

  /**
    * The message shown on the login screen above credentials.
    * This can be any informational message which you would like to see for every user before signing in.
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("auth-html-message")
  var authHtmlMessage: String = ""

}

object ClientProperties {
  val KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES: String = "client.automatically.map.all.deployables"
  val KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_INITIAL: String = "client.automatically.map.all.deployables.on.initial"
  val KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_UPDATE: String = "client.automatically.map.all.deployables.on.update"
  val KEY_CLIENT_SESSION_REMEMBER_ENABLED: String = "client.session.remember.enabled"
  val KEY_CLIENT_SESSION_TIMEOUT_MINUTES: String = "client.session.timeout.minutes"
  val KEY_MAINTENANCE_FORBIDDEN_REQUESTS: String = "maintenance.forbidden_paths"
  val KEY_IMPORTABLE_PACKAGES_PATH: String = "importable.packages.path"
  val KEY_SECURE_COOKIE_ENABLED: String = "secure.cookie.enabled"
  val KEY_UPGRADE_BATCH_SIZE: String = "upgrade.batch.size"
  val KEY_UPLOAD_FILE_BUFFER_SIZE: String = "http.upload.buffer"

  val DEFAULT_IMPORTABLE_PACKAGES_PATH: String = "importablePackages"
  var DEFAULT_MAINTENANCE_FORBIDDEN_REQUESTS: String = "deployit/task:POST" +
    ",deployit/tasks/v2:POST" +
    ",deployit/control:POST" +
    ",xldeploy/task:POST" +
    ",xldeploy/tasks/v2:POST" +
    ",xldeploy/control:POST"
  val DEFAULT_UPGRADE_BATCH_SIZE: Int = 100
  val DEFAULT_UPLOAD_FILE_BUFFER_SIZE: Int = 256 * 1024
  val KEY_HIDE_INTERNALS: String = "hide.internals"
}
