package com.xebialabs.xlplatform.scheduler

import com.xebialabs.xlplatform.scheduler.Scheduler.Messages.{CancelJob, ScheduleJob}
import com.xebialabs.xlplatform.settings.CommonSettings
import org.apache.pekko.actor._

import java.lang.Math.min
import java.time.LocalDateTime
import java.time.temporal.ChronoUnit
import scala.concurrent.duration._
import scala.language.postfixOps

object Scheduler {
  def props() = Props(classOf[Scheduler])

  // Scala scheduler uses integer internally thus limiting the maximum delay possible
  val tickDuration: Long = CommonSettings(SchedulerActorSystem.actorSystem).scheduler.tickDuration.toMillis
  val MAXIMUM_SCHEDULER_DELAY: Long = Int.MaxValue.toLong * tickDuration - 1000

  object Messages {

    case class ScheduleJob(job: Job, trigger: Trigger)

    case class CancelJob(jobId: String)

  }

}

class Scheduler extends Actor with ActorLogging {
  implicit val actorSystem = SchedulerActorSystem.actorSystem
  implicit val executionContext = actorSystem.dispatcher

  var scheduledJobs = scala.collection.mutable.Map[String, Cancellable]()

  case class TriggerFired(job: Job, trigger: Trigger, lastExection: LocalDateTime, isMultiPart: Boolean)

  override def receive: Receive = {
    case ScheduleJob(job, trigger) => scheduleJob(job, trigger)
    case CancelJob(jobId) => cancelJob(jobId)
    case TriggerFired(job, trigger, executionDate, isMultiPart) if executionDate.isAfter(now()) && isMultiPart => scheduleOnce(job, trigger, now())
    case TriggerFired(job, trigger, _, _) => executeJobAndReschedule(job, trigger, now())
  }

  private def scheduleJob(job: Job, trigger: Trigger): Unit = {
    log.debug(s"Scheduling job ${job.id} with trigger $trigger")
    scheduleOnce(job, trigger, now())
  }

  private def executeJobAndReschedule(job: Job, trigger: Trigger, lastExecution: LocalDateTime): Unit = {
    log.info(s"Trigger $trigger has fired, executing job ${job.id}")
    job.execute()
    scheduleOnce(job, trigger, lastExecution)
  }

  private def scheduleOnce(job: Job, trigger: Trigger, lastExecution: LocalDateTime): Unit = {
    val nextExecution = trigger.nextExecution(lastExecution)
    log.debug(s"Next job ${job.id} execution scheduled at: $nextExecution")
    val delay = min(ChronoUnit.MILLIS.between(now(),nextExecution), Scheduler.MAXIMUM_SCHEDULER_DELAY)
    val cancellable = context.system.scheduler.scheduleOnce(delay millis, self, TriggerFired(job, trigger, nextExecution, delay == Scheduler.MAXIMUM_SCHEDULER_DELAY))
    scheduledJobs(job.id) = cancellable
  }

  private def cancelJob(jobId: String): Unit = {
    scheduledJobs.remove(jobId) match {
      case Some(job) =>
        log.debug(s"Cancelling scheduled job $jobId")
        job.cancel()
        if (job.isCancelled) {
          log.debug(s"Scheduled job $jobId was cancelled successfully")
        } else {
          log.warning(s"Unable to cancel job $jobId. Please restart server for the changes to take effect")
        }
      case None => log.debug(s"No job $jobId currently scheduled. Ignoring...")
    }
  }

  def now(): LocalDateTime = LocalDateTime.now()

  override def postStop(): Unit = scheduledJobs.keys.foreach(cancelJob)
}