package com.xebialabs.xlplatform.scheduler

import java.util.{HashMap => JHashMap, List => JList, Map => JMap}
import akka.actor.{Actor, ActorLogging, Props}
import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, OnTaskFailurePolicy, OnTaskSuccessPolicy, Parameters}
import com.xebialabs.deployit.repository.{RepositoryService, WorkDir, WorkDirFactory}
import com.xebialabs.deployit.security.PermissionEnforcer.ROLE_ADMIN
import com.xebialabs.deployit.security.Permissions
import com.xebialabs.deployit.task.WorkdirCleanerTrigger
import com.xebialabs.xlplatform.scheduler.ControlTaskExecutor.Messages.InvokeControlTask
import com.xebialabs.xlplatform.scheduler.spring.ServiceHolder
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken
import org.springframework.security.core.authority.SimpleGrantedAuthority
import org.springframework.security.core.context.SecurityContextHolder

import scala.jdk.CollectionConverters._
import scala.concurrent.ExecutionContextExecutor
object ControlTaskExecutor {
  def props(): Props = Props(new ControlTaskExecutor())

  object Messages {

    case class InvokeControlTask(configurationItemId: String, controlTaskName: String, controlTaskParams: JMap[String, String])

  }

}

class ControlTaskExecutor extends Actor with ActorLogging with ControlTaskService with WorkDirSupport with Impersonation {
  private implicit val executionContext: ExecutionContextExecutor = context.dispatcher

  val repositoryService: RepositoryService = ServiceHolder.getRepositoryService
  val taskExecutionEngine: TaskExecutionEngine = ServiceHolder.getTaskExecutionEngine
  val workDirFactory: WorkDirFactory = ServiceHolder.getWorkDirFactory

  override def receive: Receive = {
    case m: InvokeControlTask => invokeTask(m.configurationItemId, m.controlTaskName, m.controlTaskParams)
  }

  private[scheduler] def invokeTask(configurationItemId: String, controlTaskName: String, controlTaskParams: JMap[String, String]): Unit = {
    impersonateAs("admin", roles = List(ROLE_ADMIN)) {
      val controlTask = prepareControlTask(configurationItemId, controlTaskName, Option(controlTaskParams).getOrElse(new JHashMap[String, String]()))
      withWorkDir() { implicit workDir =>
        val taskId = createTask(controlTask)
        executeTask(taskId)
      }
    }
  }
}

case class ControlTaskHolder(ci: ConfigurationItem, methodDescriptor: MethodDescriptor, parameters: Parameters)

trait ControlTaskService {
  val TASK_TYPE: String = "CONTROL"

  val repositoryService: RepositoryService
  val taskExecutionEngine: TaskExecutionEngine

  def prepareControlTask(configurationItemId: String, controlTaskName: String, controlTaskParams: JMap[String, String]): ControlTaskHolder = {
    val ci = repositoryService.read[ConfigurationItem](configurationItemId)

    val methodDescriptor = ci.getType.getDescriptor.getControlTask(controlTaskName)
    checkArgument(methodDescriptor != null, s"ConfigurationItem ${ci.getId} of type ${ci.getType} does not have a control task named $controlTaskName.")

    val parameters = Option(methodDescriptor.getParameterObjectType) match {
      case Some(paramsType) =>
        val params = paramsType.getDescriptor.newInstance("parameters").asInstanceOf[Parameters]
        controlTaskParams.asScala.foreach((params.setProperty _).tupled)
        params
      case _ => null
    }

    ControlTaskHolder(ci, methodDescriptor, parameters)
  }

  def createTask(controlTask: ControlTaskHolder)(implicit workDir: WorkDir): String = {
    val steps = controlTask.methodDescriptor.invoke[JList[Any]](controlTask.ci, controlTask.parameters).asScala.map {
      case s: Step => new TaskStep(s)
      case unknown => throw new IllegalArgumentException(s"Control tasks with step type ${unknown.getClass} are not supported, please use flow.Step type.")
    }

    val description = s"Control task [${controlTask.methodDescriptor.getName}}] for ${controlTask.ci.getId}"
    val doArchive = if ("policy" == controlTask.ci.getType.getPrefix)
      controlTask.ci.hasProperty("dryRun") && !controlTask.ci.getProperty[Boolean]("dryRun")
    else
      true
    val taskSpec = new TaskSpecification(
      description, Permissions.getAuthentication, workDir,
      BlockBuilders.steps(description, None, steps.toList).build(),
      null, false, doArchive,
      OnTaskSuccessPolicy.ARCHIVE, OnTaskFailurePolicy.CANCEL_AND_ARCHIVE)

    val metadata = taskSpec.getMetadata
    metadata.put("taskType", TASK_TYPE)
    metadata.put("taskName", controlTask.methodDescriptor.getName)
    metadata.put("taskLabel", controlTask.methodDescriptor.getLabel)
    metadata.put("controlTaskTargetCI", controlTask.ci.getId)
    if (doArchive) {
      metadata.put("controlTaskTargetInternalCI", controlTask.ci.get$internalId.toString)
      metadata.put("controlTaskTargetSecureCI", controlTask.ci.get$securedCi.toString)
      metadata.put("controlTaskTargetDirectoryRef", controlTask.ci.get$directoryReference())
    }

    taskSpec.getListeners.add(new WorkdirCleanerTrigger(workDir))

    taskExecutionEngine.register(taskSpec)
  }

  def executeTask(taskId: String): Unit = {
    taskExecutionEngine.execute(taskId)
  }

  def retrieveTask(taskId: String): TaskWithBlock = {
    taskExecutionEngine.retrieveTask(taskId)
  }

  def cancelTask(taskId: String): Unit = {
    taskExecutionEngine.cancel(taskId)
  }

  def archiveTask(taskId: String): Unit = {
    taskExecutionEngine.archive(taskId)
  }

}

trait WorkDirSupport {
  val workDirFactory: WorkDirFactory

  def withWorkDir(prefix: String = "task")(func: WorkDir => Unit): Unit = {
    implicit val workDir: WorkDir = workDirFactory.newWorkDir(prefix)
    try {
      func(workDir)
    } catch {
      case e: Exception =>
        workDir.delete()
        throw e
    }
  }
}

trait Impersonation {
  def impersonateAs(username: String, password: String = "", roles: List[String] = List())(func: => Unit): Unit = {
    val context = SecurityContextHolder.getContext
    val origAuthentication = context.getAuthentication
    try {
      context.setAuthentication(new UsernamePasswordAuthenticationToken(username, password, roles.map(new SimpleGrantedAuthority(_)).asJava))
      func
    } finally {
      context.setAuthentication(origAuthentication)
    }
  }
}
