package com.xebialabs.deployit.engine.tasker

import com.xebialabs.deployit.engine.api.execution.BlockExecutionState
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.BlockStateEvent
import com.xebialabs.deployit.engine.tasker.messages._
import org.apache.pekko.actor._

object BlockExecutingActor {

  def props(task: Task, block: CompositeBlock, ctx: TaskExecutionContext, blockRouter: SatelliteBlockRouter): Props = {
    if (block.isInstanceOf[StepBlock]) throw new IllegalArgumentException("I cannot handle Step Blocks...")
    Props(new BlockExecutingActor(task, block, ctx, blockRouter)).withDispatcher(stateManagementDispatcher)
  }

  case class BlockDone(taskId: TaskId, block: Block)

  case class BlockAlreadyDone(taskId: TaskId, block: Block)

  case class BlockStateChanged(taskId: TaskId, block: Block, oldState: BlockExecutionState, state: BlockExecutionState)

}

class BlockExecutingActor(val task: Task, val block: CompositeBlock, ctx: TaskExecutionContext, blockRouter: SatelliteBlockRouter)
  extends BaseExecutionActor with SplitModifySteps with BecomeWithMdc {

  import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
  import context._

  val taskId: String = task.getId

  private def logStateChange(newState: String): Unit =
    debug(s">> {STATE CHANGE}: BlockExecutingActor => [$newState]. Task: $taskId, block: $block.")

  def updateStateAndNotify(newState: BlockExecutionState): Unit = {
    val oldState: BlockExecutionState = block.getState()
    debug(s"updating task ${task.getId} from $oldState to $newState")
    if (oldState != newState) {
      block.newState(newState)
      val msg: BlockStateChanged = BlockStateChanged(task.getId, block, oldState, newState)
      debug(s"[${block.id}] : Sending BlockStateChanged($oldState->$newState) to ${parent.path}")
      context.system.eventStream.publish(BlockStateEvent(task, block, oldState, newState))
      parent ! msg
    } else {
      debug(s"Not sending BlockStateChanged because state is not changed. Old: $oldState, new: $newState")
    }
  }

  def receive: Receive = ReceiveWithMdc(task)(start orElse sendModifyStepsToBlocks(taskId, lookUpActor) orElse reportBlockState)

  private def reportBlockState: Receive = {
    case ReportBlockState => sender() ! BlockStateReport(taskId, block.getId(), block.getState())
  }

  private def lookUpActor(blockPath: BlockPath) = {
    block.blocks.find(_.id.isParent(blockPath)).map(createChildForBlock)
  }

  private def start: Receive = {
    logStateChange("start")
    val behaviour: Receive = {
      case Start(`taskId`, runMode) if block.getState().isReadyForExecution => block match {
        case ParallelBlock(id, _, _, blocks) =>
          info(s"[${block.id}] : Processing ParallelBlock[$id] of task [${task.getId}]")
          updateStateAndNotify(BlockExecutionState.EXECUTING)
          executeParallelBlock(blocks, runMode)
        case SerialBlock(id, _, _, blocks) =>
          info(s"[${block.id}] : Processing SerialBlock[$id] of task [${task.getId}]")
          updateStateAndNotify(BlockExecutionState.EXECUTING)
          executeSerialBlock(blocks, runMode)
      }
    }
    behaviour
  }

  def executeParallelBlock(blocks: Seq[ExecutableBlock], runMode: RunMode): Unit = {
    blocks match {
      case Seq() =>
        markAsDone()
      case _ =>
        val blocksWithActors: Seq[(ExecutableBlock, ActorRef)] =
          blocks.filterNot(_.getState() == DONE).map(b => (b, createChildForBlock(b)))
        becomeWithMdc(waitForParallelBlocksCompleted(blocksWithActors))
        blocksWithActors.foreach {
          case (_, actor) => actor ! Start(taskId, runMode)
        }
    }
  }

  def executeSerialBlock(blocks: Seq[ExecutableBlock], runMode: RunMode): Unit = {
    blocks match {
      case Seq(head, tail@_*) =>
        head.getState() match {
          case DONE =>
            executeSerialBlock(tail, runMode)
          case _ =>
            val child: ActorRef = createChildForBlock(head)
            becomeWithMdc(waitForSerialBlockCompleted(blocks, child, runMode))
            child ! Start(taskId, runMode)
        }
      case Seq() => markAsDone()
    }
  }

  private def markAsDone(): Unit = {
    block.newState(DONE)
    changeState()
  }

  def createChildForBlock(b: ExecutableBlock): ActorRef = child(b.id.toBlockId) match {
    case Some(ref) => ref
    case None => b match {
      case sb: StepBlock => createChild(blockRouter.route(task, sb, executionContext(sb)), sb.id.toBlockId)
      case cb: CompositeBlock => createChild(blockRouter.route(task, cb, executionContext(cb)), cb.id.toBlockId)
    }
  }

  private def executionContext(child: ExecutableBlock) = block match {
    case _: SerialBlock => ctx
    case _: ParallelBlock => ctx.contextFor(child)
  }

  def waitForParallelBlocksCompleted(blocks: Seq[(ExecutableBlock, ActorRef)]): Receive = {
    logStateChange("waitForParallelBlocksCompleted")
    val receive: Receive = {
      case BlockDone(`taskId`, doneBlock: ExecutableBlock) =>
        debug(s"[${block.id}] : Received BlockDone message for [${doneBlock.id}]")
        val withoutCompletedBlock: Seq[(ExecutableBlock, ActorRef)] = blocks.filterNot(_._1.id == doneBlock.id)
        updateStateAndNotify(block.determineNewState(doneBlock, withoutCompletedBlock.map(_._1)))
        if (withoutCompletedBlock.isEmpty) {
          changeState()
        } else {
          becomeWithMdc(waitForParallelBlocksCompleted(withoutCompletedBlock))
        }

      case BlockStateChanged(`taskId`, changedBlock: Block, oldState, state) =>
        debug(s"[${block.id}] : Received BlockStateChanged($oldState->$state) message for [${block.id}]")
        updateStateAndNotify(block.determineNewState(changedBlock, blocks.map(_._1)))

      case s@Stop(`taskId`) =>
        debug(s"[${block.id}] : Received Stop($taskId) message, stopping sub-block")
        blocks.foreach(t => t._2 ! s)

      case m@Abort(`taskId`) =>
        debug(s"[${block.id}] : Received [$m] message, aborting sub-block")
        blocks.foreach(t => t._2 ! m)
    }
    receive orElse reportBlockState
  }

  def waitForSerialBlockCompleted(blocks: Seq[ExecutableBlock], currentBlockActor: ActorRef, runMode: RunMode): Receive = {
    logStateChange("waitForSerialBlockCompleted")
    val receive: Receive = {
      case BlockDone(`taskId`, doneBlock: Block) if runMode == RunMode.NORMAL =>
        debug(s"[${block.id}] : Received BlockDone message for [${doneBlock.id}]")
        val withoutCompletedBlock: Seq[ExecutableBlock] = blocks.tail
        updateStateAndNotify(block.determineNewState(doneBlock, withoutCompletedBlock))
        if (withoutCompletedBlock.isEmpty || doneBlock.getState != DONE) {
          changeState()
        } else {
          executeSerialBlock(withoutCompletedBlock, runMode)
        }

      case BlockDone(`taskId`, doneBlock: Block) if runMode == RunMode.FORCE_CANCEL =>
        debug(s"[${block.id}] : Received BlockDone message for [${doneBlock.id}] (while ignoring failures)")
        if (blocks.tail.isEmpty) {
          updateStateAndNotify(if (block.blocks.exists(_.getState() == FAILED)) FAILED else doneBlock.getState)
          changeState()
        } else {
          doneBlock.getState match {
            case STOPPED | ABORTED => // we still obey STOP/ABORT commands (but not PauseSteps) while ignoring failures
              updateStateAndNotify(doneBlock.getState)
              changeState()
            case FAILED | DONE =>
              executeSerialBlock(blocks.tail, RunMode.FORCE_CANCEL)
          }
        }

      case BlockStateChanged(`taskId`, changedBlock: Block, oldState, state) =>
        debug(s"[${block.id}] : Received BlockStateChange($oldState->$state) message for [${changedBlock.id}]")
        updateStateAndNotify(block.determineNewState(changedBlock, blocks))

      case s@Stop(`taskId`) =>
        debug(s"[${block.id}] : Received Stop($taskId) message, stopping sub-block")
        currentBlockActor ! s

      case m@Abort(`taskId`) =>
        debug(s"[${block.id}] : Received [$m] message, aborting sub-block")
        currentBlockActor ! m
    }
    receive orElse reportBlockState
  }

  def changeState(): Unit = {
    if (block.getState() == DONE) {
      doNotAcceptMessages(s"[${block.id}] : All is [EXECUTED]")
    } else {
      becomeWithMdc(receive)
    }
    info(s"[${block.id}] : All sub blocks of [${block.id}] completed, notifying parent")
    parent ! BlockDone(taskId, block)
  }
}
