package com.xebialabs.deployit.engine.tasker.query

import org.apache.pekko.actor.{Actor, ActorIdentity, ActorRef, ActorSelection, DeadLetter, Identify, Props, Terminated}
import com.xebialabs.deployit.engine.tasker.query.QueryActor.messages.{QueryResponseFailure, QueryResponseSuccess}
import com.xebialabs.deployit.engine.tasker.query.QueryExecutorHandler.QueryResponsePayload
import com.xebialabs.deployit.engine.tasker.query.QueryListeningActor.ForwardQuery
import com.xebialabs.deployit.engine.tasker.query.RemoteServerQuery.{QueryFailedException, QueryNotExecutedException}

import scala.concurrent.Promise

object QueryListeningActor {

  def props(promise: Promise[QueryResponsePayload]): Props =
    Props(new QueryListeningActor(promise))

  case class ForwardQuery(mastersActorSelections: List[ActorSelection], message: AnyRef)
}

class QueryListeningActor(promise: Promise[QueryResponsePayload]) extends Actor {

  import context._

  def receive: Actor.Receive = {
    case ForwardQuery(firstMasterSelection :: restMastersSelections, message) =>
      become(identifyActor(restMastersSelections, message))
      firstMasterSelection ! Identify("")
  }

  def identifyActor(mastersActorSelections: List[ActorSelection], originalMessage: AnyRef): Actor.Receive = {
    case ActorIdentity(_, Some(actorRef)) =>
      watch(actorRef)
      system.eventStream.subscribe(self, classOf[DeadLetter])
      become(await(actorRef))
      actorRef ! originalMessage
    case _ if mastersActorSelections.nonEmpty =>
      become(identifyActor(mastersActorSelections.tail, originalMessage))
      mastersActorSelections.head ! Identify("")
    case _ =>
      promise.failure(new QueryNotExecutedException("no actor to execute query"))
  }

  def await(actorRef: ActorRef): Actor.Receive = {
    case Terminated(`actorRef`) | DeadLetter(_, `self`, `actorRef`) =>
      promise.tryFailure(new QueryNotExecutedException(s"Failed to execute [QueryRequest] because operation was terminated by a different process"))
    case QueryResponseFailure(message, exception) =>
      forget(actorRef)
      promise.tryFailure(new QueryFailedException(message, exception))
    case org.apache.pekko.actor.Status.Failure(exception) =>
      forget(actorRef)
      promise.tryFailure(new QueryFailedException(s"Failed to execute [QueryRequest] because operation failure", exception))
    case QueryResponseSuccess(response) =>
      forget(actorRef)
      promise.trySuccess(response)
  }

  private def forget(actorRef: ActorRef): Unit = {
    unwatch(actorRef)
    system.eventStream.unsubscribe(self)
  }
}
