package com.xebialabs.deployit.engine.tasker

import akka.actor.Status.Failure
import akka.actor._
import akka.event.LoggingReceive
import com.xebialabs.deployit.engine.api.execution.{BlockExecutionState, StepExecutionState}
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.BlockStateEvent
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.messages.{ExecuteStep, StopRequested}
import com.xebialabs.deployit.engine.tasker.messages._

object StepBlockExecutingActor {

  def props(task: Task, block: StepBlock, ctx: TaskExecutionContext): Props =
    Props(new StepBlockExecutingActor(task, block, ctx)).withDispatcher(stateManagementDispatcher)

}

class StepBlockExecutingActor(val task: Task, sb: StepBlock, ctx: TaskExecutionContext)
  extends BaseExecutionActor with ModifyStepsSupport with BecomeWithMdc
{
  import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
  import context._

  val block: StepBlock = sb
  val taskId: String = task.getId

  private def logStateChange(newState: String): Unit =
    debug(s">> {STATE CHANGE}: StepBlockExecutingActor => [$newState]. Task: $taskId, block: $block.")

  private val blockFinder: BlockFinder = { blockPath =>
    if (block.getId() == blockPath.toBlockId) Option(block) else None
  }

  def receive: Receive = {
    logStateChange("receive")
    LoggingReceive.withLabel("receive")(ReceiveWithMdc(task)(modifySteps(taskId, blockFinder) orElse start))
  }

  private def start: Receive = {
    logStateChange("start")
    LoggingReceive.withLabel("start") {
      case Start(`taskId`, runMode) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(block.getState()) =>
        updateStateAndNotify(BlockExecutionState.EXECUTING)
        debug(s"Create child StepExecutingActor for ${block.id}, ${task.getId}")
        executeStep(0, createChild(StepExecutingActor.props(task, ctx)), runMode)
      case Start(`taskId`, _) if block.getState() == DONE =>
        debug(s"[${block.id}] : Not Executing again as it is already [EXECUTED]")
        changeState(block.getState(), None)
      case Start(`taskId`, _) =>
        sender() ! Failure(new IllegalStateException(s"Can not execute a step block in state [${block.getState()}]"))
      case m@_ =>
        error(s"[${block.id}] : Don't know what to do with [$m]")
        sender() ! Failure(new IllegalStateException(s"$m"))
    }
  }

  def executeStep(stepNr: Int, stepActor: ActorRef, runMode: RunMode): Unit = {
    if (block.steps.size <= stepNr) {
      val finalBlockState =
        if (runMode == RunMode.FORCE_CANCEL && block.steps.exists(_.getState == StepExecutionState.FAILED)) BlockExecutionState.FAILED
        else BlockExecutionState.DONE
      changeState(finalBlockState, Option(stepActor))
    } else {
      val head = block.steps(stepNr).asInstanceOf[TaskStep]
      if (runMode == RunMode.FORCE_CANCEL && head.getState == StepExecutionState.FAILED) {
        executeStep(stepNr + 1, stepActor, runMode)
      } else {
        becomeWithMdc(waitForStep(head, stepNr + 1, stepActor, runMode))
        stepActor ! ExecuteStep(head, ctx.stepContext(head, block.id / (stepNr + 1), task))
      }
    }
  }

  def waitForStep(step: TaskStep, nextStepNr: Int, stepActor: ActorRef, runMode: RunMode): Receive = {
    logStateChange("waitForStep")
    LoggingReceive.withLabel("waitForStep") {
      case StepExecutionState.DONE | StepExecutionState.SKIPPED =>
        executeStep(nextStepNr, stepActor, runMode)
      case StepExecutionState.PAUSED => runMode match {
          case RunMode.NORMAL => changeState(BlockExecutionState.STOPPED, Option(stepActor))
          case RunMode.FORCE_CANCEL => executeStep(nextStepNr, stepActor, runMode)
        }
      case StepExecutionState.FAILED => runMode match {
          case RunMode.NORMAL => changeState(BlockExecutionState.FAILED, Option(stepActor))
          case RunMode.FORCE_CANCEL =>
            info(s"Step [${step.getDescription}] failed, but we're told to ignore failures... carry on!")
            updateStateAndNotify(BlockExecutionState.FAILING)
            executeStep(nextStepNr, stepActor, runMode)
        }
      case Stop(`taskId`) =>
        info(s"[${block.id}] : Received [Stop] message")
        updateStateAndNotify(BlockExecutionState.STOPPING)
        stepActor ! StopRequested
        becomeWithMdc(stopAfterCurrentStep(step, nextStepNr, stepActor))
      case Abort(`taskId`) =>
        info(s"[${block.id}] : Received [Abort] message")
        doAbort(stepActor, step)
    }
  }

  def doAbort(stepActor: ActorRef, step: TaskStep): Unit = {
    info(s"[${block.id}] : Received [Abort] message")
    updateStateAndNotify(BlockExecutionState.ABORTING)
    becomeWithMdc(aborting(step, stepActor))
    stepActor ! Abort(taskId)
  }

  def aborting(step: TaskStep, stepActor: ActorRef): Receive = {
    logStateChange("aborting")
    LoggingReceive.withLabel("aborting") {
      case message: StepExecutionState if message.isFinished =>
        info(s"[${block.id}] : Received [$message], going to aborted state")
        changeState(BlockExecutionState.ABORTED, Some(stepActor))
      case m@_ =>
        info(s"[${block.id}] : Received [$m], ignoring while in ABORTING state")
    }
  }

  def changeState(state: BlockExecutionState, stepActorOption: Option[ActorRef]): Unit = {
    state match {
      case DONE =>
        doNotAcceptMessages("Block done")
      case ABORTED | FAILED | STOPPED =>
        debug(s"ChangeState [$state] for [${block.id}],[${task.getId}]. Stopping Step Actor")
        stepActorOption.foreach(context.stop)
        becomeWithMdc(modifySteps(taskId, blockFinder) orElse receive)
      case _ =>
        debug(s"ChangeState [$state] for [${block.id}],[${task.getId}]. Stopping Step Actor")
        stepActorOption.foreach(context.stop)
        becomeWithMdc(start)
    }
    updateStateAndNotify(state)
    parent ! BlockDone(task.getId, block)
  }

  def stopAfterCurrentStep(step: TaskStep, nextStepNr: Int, stepActor: ActorRef): Receive = {
    logStateChange("stopAfterCurrentStep")
    LoggingReceive.withLabel("stopAfterCurrentStep") {
      case Abort(`taskId`) =>
        debug(s"StopAfterCurrentStep Received [Abort] for [${block.id}],[$taskId].")
        doAbort(stepActor, step)
      case StepExecutionState.DONE if block.steps.size <= nextStepNr =>
        changeState(BlockExecutionState.DONE, Option(stepActor))
      case StepExecutionState.FAILED =>
        changeState(BlockExecutionState.FAILED, Option(stepActor))
      case m: StepExecutionState =>
        debug(s"StopAfterCurrentStep received [$m] for [${block.id}],[$taskId]. ChangeState STOPPED")
        changeState(BlockExecutionState.STOPPED, Option(stepActor))
    }
  }

  def updateStateAndNotify(newState: BlockExecutionState): Unit = {
    val oldState: BlockExecutionState = block.getState()
    if (oldState != newState) {
      block.newState(newState)
      val msg: BlockStateChanged = BlockStateChanged(task.getId, block, oldState, newState)
      debug(s"[${block.id}] : Sending BlockStateChange($oldState->$newState) message for [${block.id}]")
      context.system.eventStream.publish(BlockStateEvent(task, block, oldState, newState))
      parent ! msg
    }
  }
}
