package com.xebialabs.deployit.engine.tasker

import akka.actor._
import akka.event.EventStream
import com.xebialabs.deployit.engine.api.execution.{FetchMode, TaskWithBlock}
import grizzled.slf4j.Logging

object TaskRegistryExtension extends ExtensionId[TaskRegistryExtension] with ExtensionIdProvider {
    //The lookup method is required by ExtensionIdProvider,
    // so we return ourselves here, this allows us
    // to configure our extension to be loaded when
    // the ActorSystem starts up
    override def lookup: ExtensionId[TaskRegistryExtension] = TaskRegistryExtension

    //This method will be called by Akka
    // to instantiate our Extension
    override def createExtension(system: ExtendedActorSystem) = new TaskRegistryExtension(system.eventStream)
}

class TaskRegistryExtension(eventStream: EventStream) extends Extension with Logging {
  info("Booting Task Registry")
  val store: Cache[String, Task] = Cache()

  def countTasks: Int = store.size

  def getFullTask(id: TaskId): Option[Task] = store.get(id)

  def getTask(id: TaskId, fetchMode: FetchMode): Option[TaskWithBlock] = {
    val task = store.get(id)
    debug(s"getTask for TaskId - $id and FetchMode- $fetchMode.")
    fetchMode match {
      case FetchMode.SUMMARY => task.map(t => new TaskSummary(t))
      case _ => task
    }
  }

  def getTasks(fetchMode: FetchMode): Seq[TaskWithBlock] = {
    val tasks = store.values.toList
    debug(s"getTasks for FetchMode- $fetchMode.")
    fetchMode match {
      case FetchMode.SUMMARY => tasks.map(t => new TaskSummary(t))
      case _ => tasks
    }
  }

  def deleteTask(id: TaskId): Unit =  {
    info(s"Delete from task registry. TaskId - $id")
    store.remove(id)
    if (store.isEmpty) {
      debug("Task registry empty.")
      eventStream.publish(TaskRegistryEmpty())
    }
  }

  protected[tasker] def clear(): Unit = {
    store.clear()
    info(s"Cleared tasks from task registry")
  }

  def register(task: Task): Unit = {
    info(s"register Task - ${task.getId}")
    if (store.contains(task.getId)) {
      error(s"Task with id [${task.getId}] is already registered")
      throw new IllegalStateException(s"Task with id [${task.getId}] is already registered")
    }
    store.put(task.getId, task)
  }
}

case class TaskRegistryEmpty()
