package com.xebialabs.deployit.engine.tasker

import akka.actor.{ActorRef, Cancellable, PoisonPill, Props}
import akka.event.LoggingReceive
import com.github.nscala_time.time.Imports._
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState.{CANCELLED, SCHEDULED}
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.{Cancel, Schedule}
import com.xebialabs.deployit.engine.tasker.messages.{Cancelled, Enqueue, TaskDone, TaskStateEventHandled}
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTaskRepository, PendingTaskRepository}
import com.xebialabs.xlplatform.settings.CommonSettings
import org.joda.time.DateTime

import scala.concurrent.duration._

object TaskSchedulingActor {
  def name(taskId: TaskId) = s"task-scheduling-actor-$taskId"

  def props(taskId: TaskId, taskRepository: ActiveTaskRepository, pendingTaskRepository: PendingTaskRepository,
            taskQueueService: TaskQueueService, distributor: ActorRef): Props =
    Props(new TaskSchedulingActor(taskId, taskRepository, pendingTaskRepository, taskQueueService, distributor)).withDispatcher(stateManagementDispatcher)
}

class TaskSchedulingActor(val taskId: TaskId, val taskRepository: ActiveTaskRepository,
                          val pendingTaskRepository: PendingTaskRepository,
                          val taskQueueService: TaskQueueService, val distributor: ActorRef)
  extends BaseExecutionActor with ModifyStepsSupport with UpdateStateSupport with TaskActorSupport {

  import context._

  protected val commonSettings: CommonSettings = CommonSettings(system)
  override val task: Task = pendingTaskRepository.task(taskId, loadFullSpec = true).flatMap(_.spec).map(spec => new Task(taskId, spec)).orNull
  registerStateListeners(task)

  case object ScheduledStart

  override def receive: Receive = {
    case Schedule(`taskId`, scheduleAt) =>
      logger.info(s"Schedule task [$taskId] at $scheduleAt")
      pendingTaskRepository.schedule(taskId, scheduleAt)
      doSchedule(scheduleAt, createOrLookupChildForTaskBlock())
    case Cancel(`taskId`, notificationActor, _) =>
      doCancelWhenPending(Some(notificationActor))
  }

  def calculateDelay(time: DateTime): FiniteDuration = FiniteDuration((DateTime.now to time).millis, MILLISECONDS)

  def doSchedule(scheduleAt: DateTime, blockActor: ActorRef): Unit = {
    info(s"Received [Schedule] message for task [$taskId]")
    task.setScheduledDate(scheduleAt)
    updateStateAndNotify(SCHEDULED)

    val delay: FiniteDuration = calculateDelay(scheduleAt)
    info(s"Going to schedule task [$taskId] at [${scheduleAt.toString("yyyy-MM-dd hh:mm:ss Z")}] which is [$delay] from now")

    if (delay.toMinutes < 0) {
      prepareAndEnqueueTask(taskId)
      self ! PoisonPill
    } else {
      val scheduleHandle: Cancellable = context.system.scheduler.scheduleOnce(delay, self, ScheduledStart)
      becomeWithMdc(scheduled(scheduleHandle))
    }
  }

  def scheduled(scheduleHandle: Cancellable): Receive = {
    logStateChange("scheduled")
    LoggingReceive.withLabel("scheduled") {
      case Enqueue(`taskId`) =>
        scheduleHandle.cancel()
        prepareAndEnqueueTask(taskId)
        self ! PoisonPill
      case Schedule(`taskId`, scheduleAt) =>
        scheduleHandle.cancel()
        doSchedule(scheduleAt, createOrLookupChildForTaskBlock())
      case ScheduledStart =>
        prepareAndEnqueueTask(taskId)
        self ! PoisonPill
      case Cancel(`taskId`, notificationActor, _) if task.getStartDate == null =>
        scheduleHandle.cancel()
        doCancelWhenPending(Some(notificationActor))
      case _: TaskStateEventHandled =>
      case msg: AnyRef =>
        throw new IllegalStateException(msg.toString)
    }
  }

  def doCancelWhenPending(notificationActor: Option[ActorRef]): Unit = {
    info(s"Received [Cancel] message for task [$taskId]")
    updateStateAndNotify(CANCELLED)
    notifyTaskDone()
    TaskRegistryExtension(system).deleteTask(taskId)
    becomeWithMdc(cancelled(notificationActor))
  }

  private def cancelled(notificationActor: Option[ActorRef]): Receive = {
    case TaskStateEventHandled(`taskId`, _, CANCELLED) =>
      notificationActor.foreach {
        _ ! Cancelled(taskId)
      }
      self ! PoisonPill
    case _@message =>
      warn(s"warn message: {$message} while waiting for cancelled state event")

  }

  def notifyTaskDone(): Unit = {
    info(s"Task [$taskId] is completed with state [${task.getState}]")
    context.system.eventStream.publish(TaskDone(task))
  }

}

object TaskScheduler {
  def name = "task-scheduler"

  def props(taskRepository: ActiveTaskRepository, pendingTaskRepository: PendingTaskRepository,
            taskQueueService: TaskQueueService, distributor: ActorRef): Props =
    Props(new TaskScheduler(taskRepository, pendingTaskRepository, taskQueueService, distributor))
}

class TaskScheduler(taskRepository: ActiveTaskRepository, pendingTaskRepository: PendingTaskRepository,
                    taskQueueService: TaskQueueService, distributor: ActorRef) extends BaseExecutionActor {

  import context._

  override def receive: Receive = {
    case msg@Schedule(taskId, scheduleAt) =>
      logger.info(s"Schedule task [$taskId] at $scheduleAt")
      pendingTaskRepository.schedule(taskId, scheduleAt)
      createOrLookupChildForScheduledTask(taskId).forward(msg)
    case msg@Cancel(taskId, _, _) =>
      logger.info(s"Cancel task [$taskId]")
      createOrLookupChildForScheduledTask(taskId).forward(msg)
  }

  def createOrLookupChildForScheduledTask(taskId: TaskId): ActorRef = {
    val actorName = TaskSchedulingActor.name(taskId)
    child(actorName) match {
      case Some(ref) =>
        ref
      case None =>
        createChild(
          TaskSchedulingActor.props(taskId, taskRepository, pendingTaskRepository, taskQueueService, distributor), actorName)
    }
  }

}
