package com.xebialabs.deployit.engine.tasker

import java.io.{File, FileFilter}
import java.nio.file.Files

import akka.actor._
import akka.pattern._
import akka.util.Timeout
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.spi.services.RepositoryFactory
import com.xebialabs.deployit.engine.tasker.RecoverySupervisorActor.{ReadAll, Tasks}
import com.xebialabs.deployit.engine.tasker.TaskFinalizer._
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Recovered
import com.xebialabs.deployit.engine.tasker.distribution._
import com.xebialabs.deployit.engine.tasker.distribution.versioning.ConfigurationHashProvider
import com.xebialabs.deployit.engine.tasker.log.{DefaultStepLogFactory, DefaultStepLogRetriever, StepLogFactory, StepLogRetriever}
import com.xebialabs.deployit.engine.tasker.query.RemoteServerQuery
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTaskRepository, PendingTaskRepository}
import com.xebialabs.xlplatform.settings.shared.TaskerSettings
import grizzled.slf4j.Logging
import org.springframework.jms.config.JmsListenerContainerFactory
import org.springframework.jms.listener.DefaultMessageListenerContainer

import scala.concurrent.Await
import scala.util.Try

object TaskExecutionWorker extends Logging {

  def initialize(masters: Seq[String],
                 hostnameToActorPathTransformer: String => String,
                 repositoryFactory: RepositoryFactory,
                 stepLogFactory: StepLogFactory,
                 stepLogRetriever: StepLogRetriever,
                 archive: Archive,
                 workerName: Option[String],
                 workerPublicKey: Array[Byte],
                 configurationHashProvider: ConfigurationHashProvider,
                 activeTaskRepository: ActiveTaskRepository,
                 pendingTaskRepository: PendingTaskRepository,
                 workerRepository: TaskExecutionWorkerRepository,
                 containerFactory: JmsListenerContainerFactory[DefaultMessageListenerContainer],
                 taskerSettings: TaskerSettings,
                 shouldNotChunk: Boolean,
                 shutdown: () => Unit = () => TaskExecutionWorker.this.shutdownSystem())(implicit system: ActorSystem): TaskRecoverer = {
    ActiveTasksQueue(system)

    val taskRegistry = TaskRegistryExtension(system)
    val archiver: ActorRef = system.actorOf(ArchiveActor.props(archive, stepLogRetriever), ArchiveActor.name)
    val defaultAddress = system.asInstanceOf[ExtendedActorSystem].provider.getDefaultAddress
    val detectedWorkerName = workerName.getOrElse(Seq(defaultAddress.host, defaultAddress.port).flatten.mkString(":"))
    val workerReceiverProps = WorkerReceiver.props(masters, hostnameToActorPathTransformer.andThen(_ + WorkerManager.name),
      detectedWorkerName, workerPublicKey, configurationHashProvider.configurationHash, shutdown,
      () => taskRegistry.countTasks == 0)
    system.actorOf(workerReceiverProps, WorkerReceiver.name)

    RemoteServerQuery(masters, hostnameToActorPathTransformer)

    val taskActorCreator: TaskActorCreator = new TaskActorCreatorImpl(repositoryFactory, stepLogFactory, archiver, activeTaskRepository)
    val taskManagerProps = TasksManager.props(taskActorCreator, taskRegistry, activeTaskRepository, shouldNotChunk)
    val tasksManager: ActorRef = system.actorOf(taskManagerProps, TasksManager.name)
    val recoveryProps = RecoverySupervisorActor.props(taskerSettings.recoveryDir)

    val recoverySupervisor: ActorRef = system.actorOf(recoveryProps, RecoverySupervisorActor.name)
    system.actorOf(TaskRecoveryListener.props(recoverySupervisor), TaskRecoveryListener.name)

    val taskFinalizer: TaskFinalizer = new TaskFinalizerImpl(system, activeTaskRepository, pendingTaskRepository,
      taskerSettings.askTimeout)
    system.actorOf(TaskTransitionActor.props(taskFinalizer, tasksManager), TaskTransitionActor.name)

    val workerAddress = if (system.settings.config.getBoolean("deploy.task.in-process-worker")) {
      system.toString
    } else {
      defaultAddress.toString
    }
    val taskReceiver: ActorRef = system.actorOf(TaskReceiver.props(workerAddress, activeTaskRepository, pendingTaskRepository,
      workerRepository, tasksManager), TaskReceiver.name)
    EnqueueTaskListener.registerTaskListenerContainers(workerAddress, taskReceiver, containerFactory, system.settings.config)
    new TaskRecoverer(taskerSettings, recoverySupervisor, tasksManager, taskFinalizer, stepLogFactory)
  }

  private def shutdownSystem(): Unit = {
    System.exit(0)
  }

  val inProcessWorkerName = "In-process worker"
  val localMasterAddress = s"/user/"

  def local(repositoryFactory: RepositoryFactory, archive: Archive, configurationHashProvider: ConfigurationHashProvider,
            taskRepository: ActiveTaskRepository, pendingTaskRepository: PendingTaskRepository,
            workerRepository: TaskExecutionWorkerRepository,
            containerFactory: JmsListenerContainerFactory[DefaultMessageListenerContainer],
            taskerSettings: TaskerSettings, stepLogFactory: StepLogFactory = new DefaultStepLogFactory,
            stepLogRetriever: StepLogRetriever = new DefaultStepLogRetriever)
           (implicit system: ActorSystem): TaskRecoverer =
    initialize(Seq(localMasterAddress), Predef.identity, repositoryFactory, stepLogFactory, stepLogRetriever, archive, Some(inProcessWorkerName),
      workerPublicKey = null, configurationHashProvider = configurationHashProvider, activeTaskRepository = taskRepository,
      pendingTaskRepository = pendingTaskRepository, workerRepository = workerRepository, containerFactory = containerFactory,
      taskerSettings = taskerSettings, shouldNotChunk = true,
      shutdown = () => {
        warn("Attempt to shutdown local worker.")
      })

  class TaskRecoverer(taskerSettings: TaskerSettings,
                      recoverySupervisor: ActorRef,
                      tasksManager: ActorRef,
                      taskFinalizer: TaskFinalizer,
                      stepLogFactory: StepLogFactory)
                     (implicit system: ActorSystem) {
    def recoverTasks(): Unit = {
      cleanUpTmpFiles()
      try {
        implicit val timeout: Timeout = taskerSettings.askTimeout
        val tasks = Await.result(recoverySupervisor ? ReadAll, timeout.duration).asInstanceOf[Tasks].tasks
        for (task <- tasks) Try(onRecover(task)).recover {
          case e: Exception => error("Error while recovering tasks.", e)
        }
      } catch {
        case e: Exception => error("Error while recovering.", e)
      }
    }

    private def cleanUpTmpFiles(): Unit = {
      val tmpFiles = taskerSettings.recoveryDir.listFiles(new FileFilter {
        override def accept(pathname: File): Boolean = pathname.getName.endsWith(".task.tmp")
      })
      Option(tmpFiles).toList.flatten.foreach { file =>
        try {
          warn(s"Deleting corrupted task file: ${file.getAbsolutePath}")
          Files.delete(file.toPath)
        } catch {
          case e: Throwable => error(s"Unable to delete corrupted recovery file: ${file.getAbsolutePath}", e)
        }
      }
    }

    def onRecover(task: Task): Unit = {
      implicit val timeout: Timeout = taskerSettings.askTimeout
      task.context.stepLogFactory = stepLogFactory
      task.context.childContexts.foreach(c => c._2.stepLogFactory = stepLogFactory)
      Await.ready(tasksManager ? Recovered(task), timeout.duration)
      if (task.cancelling) {
        taskFinalizer.cancel(findActor(tasksManager, task.getId), task.getId)
      }
    }
  }

}
