package com.xebialabs.deployit.engine.tasker.query

import akka.actor.{ActorRef, ActorSelection, ActorSystem}
import com.google.common.cache.{Cache, CacheBuilder}
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.engine.tasker.query.QueryActor.messages.QueryRequest
import com.xebialabs.deployit.engine.tasker.query.QueryExecutorHandler.{QueryPayload, QueryResponsePayload}
import com.xebialabs.deployit.engine.tasker.query.QueryListeningActor.ForwardQuery
import com.xebialabs.deployit.engine.tasker.query.RemoteServerQuery.{QueryFailedException, QueryTimeoutException}
import grizzled.slf4j.Logging

import java.time
import java.util.concurrent.atomic.AtomicReference
import scala.concurrent.duration.{FiniteDuration, _}
import scala.concurrent.{Await, Promise, TimeoutException}
import scala.util.{Failure, Random, Success}

object RemoteServerQuery {

  class QueryNotExecutedException(message: String) extends DeployitException(message)

  class QueryFailedException(message: String, ex: Throwable) extends DeployitException(ex, message)

  class QueryTimeoutException(message: String, ex: Throwable) extends DeployitException(ex, message)

  private val remoteInstance: AtomicReference[RemoteServerQuery] = new AtomicReference[RemoteServerQuery]()
  private val cachedInstance: AtomicReference[CachedRemoteServerQueryImpl] = new AtomicReference[CachedRemoteServerQueryImpl]()

  def apply(masters: Seq[String], hostnameToActorPathTransformer: String => String)(implicit system: ActorSystem): RemoteServerQuery = {
    val mastersActorSelections = masters.map(queryActorSelection(_, hostnameToActorPathTransformer))
    val remoteServerQueryImpl = new RemoteServerQueryImpl(mastersActorSelections.toList)
    cachedInstance.updateAndGet(oldCached =>
      new CachedRemoteServerQueryImpl(remoteServerQueryImpl, Option(oldCached).map(_.cachedData))
    )
    remoteInstance.updateAndGet(_ => remoteServerQueryImpl)
  }

  def apply(cache: Boolean = false): RemoteServerQuery = {
    val instance = if (cache) cachedInstance.get else remoteInstance.get
    Option(instance)
      .getOrElse(throw new IllegalStateException("RemoteServerQuery instance not initialized"))
  }

  def instance(): RemoteServerQuery = remoteInstance.get()

  def queryActorSelection(master: String, hostnameToActorPathTransformer: String => String)(implicit system: ActorSystem): ActorSelection =
    system.actorSelection(s"${hostnameToActorPathTransformer(master)}/${QueryActor.name}")
}

trait RemoteServerQuery {
  def query(query: QueryPayload): QueryResponsePayload

  def mastersActorSelections(): List[ActorSelection]
}

private class CachedRemoteServerQueryImpl(val delegate: RemoteServerQuery,
                                          oldCache: Option[Cache[String, QueryResponsePayload]],
                                          val timeout: FiniteDuration = 1.minute,
                                          val maximumSize: Int = 10000)
  extends RemoteServerQuery with Logging {

  lazy val cachedData: Cache[String, QueryResponsePayload] = oldCache.getOrElse {
        CacheBuilder.newBuilder()
          .asInstanceOf[CacheBuilder[String, QueryResponsePayload]]
          .expireAfterWrite(time.Duration.ofSeconds(timeout.toSeconds))
          .maximumSize(maximumSize)
          .build()
  }

  override def query(query: QueryPayload): QueryResponsePayload = cachedData.get(query.getClass.getName + query.key(), () => delegate.query(query))

  override def mastersActorSelections(): List[ActorSelection] = delegate.mastersActorSelections()
}

private class RemoteServerQueryImpl(val mastersActorSelections: List[ActorSelection], val timeout: FiniteDuration = 1.minute)(implicit val system: ActorSystem) extends RemoteServerQuery with Logging {

  override def query(query: QueryPayload): QueryResponsePayload =
    send(query => ForwardQuery(Random.shuffle(mastersActorSelections), query), query, QueryRequest)

  private def send(forward: AnyRef => AnyRef, query: QueryPayload, toMessage: (QueryPayload, ActorRef) => AnyRef): QueryResponsePayload = {
    val p = Promise[QueryResponsePayload]
    val listener: ActorRef = system.actorOf(QueryListeningActor.props(p))
    val message: AnyRef = toMessage(query, listener)
    listener ! forward(message)
    try {
      Await.ready(p.future, timeout)
      p.future.value.get match {
        case Success(responsePayload) =>
          debug(s"Query [$query] successfully executed.")
          responsePayload
        case Failure(ex) => throw ex
      }
    } catch {
      case ex: TimeoutException => throw new QueryTimeoutException("Timeout during query execution", ex)
      case ex: InterruptedException => throw new QueryFailedException("Interruption during query execution", ex)
    } finally {
      system.stop(listener)
    }
  }
}

object RemoteServerQueryHandler {
  def query(query: QueryPayload, cache: Boolean = false): QueryResponsePayload = RemoteServerQuery(cache).query(query)
}

