package com.xebialabs.deployit.engine.tasker

import akka.actor.{Actor, ActorRef, Props}
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState._
import com.xebialabs.deployit.engine.tasker.TaskFinalizer._
import com.xebialabs.deployit.engine.tasker.messages.TaskDone
import com.xebialabs.deployit.plugin.api.udm.{OnTaskFailurePolicy, OnTaskSuccessPolicy}
import grizzled.slf4j.Logging

object TaskTransitionActor {
  def props(finalizer: TaskFinalizer, tasksManager: ActorRef): Props =
    Props(new TaskTransitionActor(finalizer, tasksManager)).withDispatcher(stateManagementDispatcher)

  val name = "task-transition-actor"
  val CONTROL_TASK = "CONTROL"

}

class TaskTransitionActor(finalizer: TaskFinalizer, tasksManager: ActorRef) extends Actor with Logging {

  import TaskTransitionActor._
  import context._

  override def preStart(): Unit = {
    system.eventStream.subscribe(self, classOf[TaskDone])
    super.preStart()
  }

  private def taskMatches(validState: TaskExecutionState => Boolean,
                          onSuccessPolicy: Option[OnTaskSuccessPolicy] = None,
                          onFailurePolicy: Option[OnTaskFailurePolicy] = None,
                          taskTypeFilter: Option[String => Boolean] = None)(task: Task) = {
    val spec = task.getSpecification
    val matchesSuccessPolicy = onSuccessPolicy.isEmpty || onSuccessPolicy.contains(spec.getOnSuccessPolicy)
    val matchesFailurePolicy = onFailurePolicy.isEmpty || onFailurePolicy.contains(spec.getOnFailurePolicy)
    val validTaskType = taskTypeFilter.forall(_.apply(spec.getMetadata.get("taskType")))
    validTaskType && matchesSuccessPolicy && matchesFailurePolicy && validState(task.getState) && !spec.isRollback
  }

  private val autoArchiveTask: Receive = {
    case TaskDone(task) if taskMatches(
      validState = _ == EXECUTED,
      onSuccessPolicy = Option(OnTaskSuccessPolicy.ARCHIVE)
    )(task) =>
      info(
        s"""Automatically transition task [${task.getId}] to DONE state because it's on success policy is [${task.getSpecification.getOnSuccessPolicy}]""".stripMargin
      )
      finalizer.archive(findActor(tasksManager, task.getId), task.getId)
  }

  private val autoCancelTask: Receive = {
    case TaskDone(task) if taskMatches(
      validState = _.isExecutionHalted,
      onFailurePolicy = Option(OnTaskFailurePolicy.CANCEL_AND_ARCHIVE)
    )(task) =>
      info(
        s"""Automatically cancelling task [${task.getId}] because it is halted and on failure policy is [${task.getSpecification.getOnFailurePolicy}]""".stripMargin
      )
      finalizer.cancel(findActor(tasksManager, task.getId), task.getId)
  }

  private val doNothing: Receive = {
    case m =>
      debug(s"Ignoring message $m")
  }

  def receive: Actor.Receive = ReceiveWithMdc()(
    autoArchiveTask.orElse(autoCancelTask).orElse(doNothing)
  )
}
