package com.xebialabs.deployit.engine.tasker

import akka.actor.{ActorRef, ActorSelection, ActorSystem}
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.ArchivedListeningActor.{Forward, ForwardToRef}
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.{ArchiveTask, Cancel}
import com.xebialabs.deployit.engine.tasker.messages.RunMode
import com.xebialabs.deployit.engine.tasker.repository.{CrudTaskRepository, PendingTaskRepository}
import grizzled.slf4j.Logging

import scala.concurrent.duration.Duration
import scala.concurrent.{Await, Promise}
import scala.util.{Failure, Success}

object TaskFinalizer {
  def findActor(tasksManager: ActorRef, taskId: TaskId)(implicit system: ActorSystem): ActorSelection = {
    system.actorSelection(tasksManager.path / taskId)
  }
}

trait TaskFinalizer {
  def archive(actor: ActorSelection, taskId: TaskId): Unit
  def cancel(actor: ActorSelection, taskId: TaskId, runMode: RunMode = RunMode.NORMAL): Unit
  def cancelPendingTask(actor: ActorRef, taskId: TaskId, runMode: RunMode = RunMode.NORMAL): Unit
}

class TaskFinalizerImpl(system: ActorSystem, taskRepository: CrudTaskRepository, pendingTaskRepository: PendingTaskRepository, implicit val askTimeout: Timeout) extends TaskFinalizer with Logging {
  override def archive(actor: ActorSelection, taskId: TaskId): Unit = {
    send(msg => Forward(actor, msg), taskId, ArchiveTask, taskRepository.delete)
  }

  override def cancel(actor: ActorSelection, taskId: TaskId, runMode: RunMode = RunMode.NORMAL): Unit = {
    send(msg => Forward(actor, msg), taskId, Cancel(_, _, runMode), taskRepository.delete)
  }

  override def cancelPendingTask(actor: ActorRef, taskId: TaskId, runMode: RunMode = RunMode.NORMAL): Unit = {
    send(msg => ForwardToRef(actor, msg), taskId, Cancel(_, _, runMode), pendingTaskRepository.delete)
  }

  private def send(forward: AnyRef => AnyRef, taskId: TaskId, msg: (TaskId, ActorRef) => AnyRef, deleteTask: String => Unit): Unit = {
    val p = Promise[TaskId]()
    val listener: ActorRef = system.actorOf(ArchivedListeningActor.props(taskId, p))
    val message: AnyRef = msg(taskId, listener)
    listener ! forward(message)
    Await.ready(p.future, Duration.Inf)
    system.stop(listener)
    p.future.value.get match {
      case Failure(ex) => throw ex
      case Success(taskId) =>
        deleteTask(taskId)
        debug(s"Task $taskId successfully archived.")
    }
  }
}
