package com.xebialabs.deployit.engine.tasker;

import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.plugin.api.flow.Step;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.*;
import static org.joda.time.DateTimeZone.UTC;

@SuppressWarnings("serial")
public class TaskStep implements StepState, Serializable {

    private final Step implementation;
    private final Map<String, String> metadata = new HashMap<>();
    private final boolean skippable;
    private volatile StepExecutionState state;
    private transient Thread runningThread;
    private final AtomicInteger failureCount;
    private List<String> previousAttemptsLogs = new ArrayList<>();
    private StringBuilder logBuilder;
    private DateTime startDate;
    private DateTime completionDate;
    private DateTime lastModificationDate;
    private final String description;

    public TaskStep(Step step) {
        this.implementation = step;
        this.state = StepExecutionState.PENDING;
        this.failureCount = new AtomicInteger();
        this.logBuilder = new StringBuilder();
        this.description = step.getDescription();
        this.skippable = step.skippable();
    }

    @SuppressWarnings("rawtypes")
    public Step getImplementation() {
        return implementation;
    }

    void setState(StepExecutionState state) {
        this.state = state;
        if (state == FAILED) {
            failureCount.incrementAndGet();
        }
        touch();
    }

    void recordStart() {
        this.startDate = new DateTime(UTC);
        touch();
    }

    void recordCompletion() {
        this.completionDate = new DateTime(UTC);
        touch();
    }

    public synchronized void markNextAttempt() {
        if (getFailureCount() > 0 && getState() == FAILED) {
            previousAttemptsLogs.add(logBuilder.toString());
            logBuilder = new StringBuilder();
        }
    }

    void touch() {
        this.lastModificationDate = new DateTime(UTC);
    }

    public boolean isMarkedForSkip() {
        return getState() == SKIP;
    }

    public boolean canSkip() {
        return this.skippable && (getState() == PENDING || getState() == FAILED || getState() == SKIP);
    }

    public boolean hasExecuted() {
        return state == StepExecutionState.DONE || state == StepExecutionState.SKIPPED;
    }

    @Override
    public String toString() {
        return "[" + description + "]";
    }

    @Override
    public StepExecutionState getState() {
        return state;
    }

    @Override
    public synchronized String getLog() {
        return logBuilder.toString();
    }

    @Override
    public List<String> getPreviousAttemptsLogs() {
        return new ArrayList<>(previousAttemptsLogs);
    }

    public void setLogs(List<String> previous, String last) {
        this.previousAttemptsLogs = previous;
        this.logBuilder = new StringBuilder(last);
    }

    public synchronized TaskStep appendLog(String stepLogMsg) {
        logBuilder.append(stepLogMsg);
        return this;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public DateTime getStartDate() {
        return startDate;
    }

    @Override
    public DateTime getCompletionDate() {
        return completionDate;
    }

    public DateTime getLastModificationDate() {
        return lastModificationDate;
    }

    @Override
    public int getFailureCount() {
        return failureCount.intValue();
    }

    public void registerCurrentThread() {
        runningThread = Thread.currentThread();
        logger.debug("Setting the thread {}", runningThread.getId());
    }

    public void interruptRunner() {
        logger.debug("Trying to interrupt task step {}", description);
        if (runningThread != null) {
            logger.debug("Interrupting the running thread {}", runningThread.getId());
            runningThread.interrupt();
        }
    }

    @Override
    public Map<String, String> getMetadata() {
        return metadata;
    }

    @Override
    public boolean getSkippable() {
        return this.skippable;
    }

    public boolean isFailed() {
        return getState() == FAILED;
    }

    public TaskStep recovered() {
        if (state == EXECUTING) {
            setState(FAILED);
        }
        return this;
    }

    static final Logger logger = LoggerFactory.getLogger(TaskStep.class);
}
