package com.xebialabs.deployit.engine.tasker

import akka.actor.{ActorRef, Props}
import akka.event.LoggingReceive
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockAlreadyDone, BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.PhaseContainerExecutingActor.internalMessages.{PhaseCompleted, ProcessRemainingPhases}
import com.xebialabs.deployit.engine.tasker.messages._

object PhaseContainerExecutingActor {
  def props(task: Task, ctx: TaskExecutionContext, parentRef: ActorRef): Props = {
    Props(new PhaseContainerExecutingActor(task, ctx, parentRef)).withDispatcher(stateManagementDispatcher)
  }

  object internalMessages {

    case class PhaseCompleted(phase: Phase)

    case class ProcessRemainingPhases(remainingPhases: Seq[Phase])

  }

}

class PhaseContainerExecutingActor(val task: Task, ctx: TaskExecutionContext, parentRef: ActorRef) extends BaseExecutionActor with ActorCreationSupport with SplitModifySteps with BecomeWithMdc {
  val taskId: String = task.getId
  val container: PhaseContainer = task.block

  private def logStateChange(newState: String): Unit =
    debug(s">> {STATE CHANGE}: PhaseContainerExecutingActor => [$newState]. Task: $taskId")

  override def receive: Receive = {
    logStateChange("receive")
    LoggingReceive.withLabel("receive")(sendModifyStepsToBlocks(taskId, lookUpPhaseActor) orElse ReceiveWithMdc(task) {
      case Start(`taskId`, runMode) if container.getState().isReadyForExecution =>
        info(s"[${container.id}] : Processing PhaseContainer of task [${task.getId}]")
        updateStateAndNotify(EXECUTING)
        executePhases(container.phases, runMode)

      case Start(`taskId`, _) =>
        error(s"can't start a container in state ${container.getState()}")

      case finishUp@FinishUp(`taskId`, runMode) =>
        debug(s"[${container.id}] : Received [$finishUp] message - going to execute finish up phases")

        val alwaysExecutedPhases = container.phases
          .collect { case Phase.AlwaysExecuted(forcedPhase) => forcedPhase }
          .dropWhile(phase => phase.getState() == DONE)
          .toList

        debug(s"Always executed phases for task $taskId are $alwaysExecutedPhases")

        alwaysExecutedPhases match {
          case Nil =>
            updateStateAndNotify(DONE)
            notifyParent()
          case _ =>
            task.cancelling = true
            updateStateAndNotify(EXECUTING)
            debug(s"Task ${task.getId} is marked as cancelled")
            executePhases(alwaysExecutedPhases, runMode)
        }
    })
  }

  private def lookUpPhaseActor(blockPath: BlockPath): Option[ActorRef] = {
    container.phases.find(_.id.isParent(blockPath)).map(p => createOrLookUpPhaseActor(p))
  }

  def updateStateAndNotify(newState: BlockExecutionState): Unit = {
    val oldState = container.getState()
    debug(s"Checking to update state from $oldState -> $newState")
    if (oldState != newState) {
      container.newState(newState)
      val msg = BlockStateChanged(task.getId, container, oldState, newState)
      debug(s"[${container.id}] : Sending BlockStateChanged($oldState->$newState) to ${parentRef.path}")
      parentRef ! msg
    } else {
      debug(s"Not sending BlockStateChanged back to parent because state didnt change. Old: $oldState, new: $newState.")
    }
  }

  def executePhases(phases: Seq[Phase], runMode: RunMode): Unit = {
    val phase = phases.head
    phase.getState() match {
      case DONE =>
        executePhases(phases.tail, runMode)
      case _ =>
        val phaseActor = createOrLookUpPhaseActor(phase)
        info(s"Sending [Start] to phase actor for phase ${phase.id}")
        phaseActor ! Start(taskId, runMode)
        becomeWithMdc(waitForPhaseCompletion(phase, phases, phaseActor, runMode))
    }
  }

  def createOrLookUpPhaseActor(phase: Phase): ActorRef =
    context.child(phase.getId()) match {
      case Some(ref) =>
        ref
      case None =>
        createChild(PhaseExecutingActor.props(task, phase, ctx), phase.getId())
    }

  def waitForPhaseCompletion(phase: Phase, phases: Seq[Phase], phaseActor: ActorRef, runMode: RunMode): Receive = {
    logStateChange(s"waitForPhaseCompletion (${phase.id.toBlockId})")
    LoggingReceive.withLabel("waitForPhaseCompletion") {
      case m@BlockDone(`taskId`, `phase`) if container.getState() == STOPPING =>
        debug(s"[${container.id}] : Received $m message for [${phase.id}]")
        val state = if (phases.tail.nonEmpty && phase.getState() == DONE) STOPPED else phase.getState()
        updateStateAndNotify(state)
        notifyParent()

      case m@BlockDone(`taskId`, `phase`) =>
        debug(s"[${container.id}] : Received $m message for [${phase.id}]")
        parentRef ! PhaseCompleted(phase)
        executeRemainingPhases(phase, phases)

      case ProcessRemainingPhases(remainingPhases) =>
        executePhases(remainingPhases, runMode)

      case m@BlockStateChanged(`taskId`, `phase`, _, _) =>
        debug(s"[${container.id}] : Received $m message for [${phase.id}]")
        val newState: BlockExecutionState = container.determineNewState(phase, phases)
        debug(s"Determined new container state to be [$newState] for $phase and $phases")
        updateStateAndNotify(newState)

      case stop@Stop(`taskId`) =>
        debug(s"[${container.id}] : Received Stop($taskId) message, stopping sub-block")
        phaseActor ! stop

      case abort@Abort(`taskId`) if phase.getState() == DONE && phases.tail.nonEmpty =>
        debug(s"[${container.id}] : Received [$abort] message, but phase ${phase.id} is already in DONE state so directly updating state to STOPPED")
        updateStateAndNotify(STOPPING)
        updateStateAndNotify(STOPPED)
        notifyParent()

      case abort@Abort(`taskId`) =>
        debug(s"[${container.id}] : Received [$abort] message, aborting sub-block")
        phaseActor ! abort
    }
  }

  def executeRemainingPhases(phase: Phase, phases: Seq[Phase]): Unit = {
    val remainingPhases = phases.tail
    val newState: BlockExecutionState = container.determineNewState(phase, remainingPhases)
    debug(s"Determined new container state to be [$newState] for $phase and $remainingPhases")
    updateStateAndNotify(newState)
    if (newState.isFinished) {
      notifyParent()
    } else {
      self ! ProcessRemainingPhases(remainingPhases)
    }
  }

  def blockDone(): Receive = {
    logStateChange("blockDone")
    LoggingReceive.withLabel("blockDone") {
      case finishUp@FinishUp(`taskId`, _) if parentRef == sender() =>
        debug(s"[${container.id}] : Received [$finishUp] message - already finished")
        parentRef ! BlockAlreadyDone(taskId, container)
      case m =>
        context.system.deadLetters forward m
    }
  }

  def notifyParent(): Unit = {
    if (container.getState() != DONE) {
      becomeWithMdc(receive)
    } else {
      becomeWithMdc(blockDone())
    }
    info(s"[${container.id}] : All phases of [${container.id}] completed, notifying parent ($parentRef")
    parentRef ! BlockDone(taskId, container)
  }
}
