package com.xebialabs.deployit.engine.tasker

import akka.actor.Status.Failure
import akka.actor._
import akka.event.LoggingReceive
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState.DONE
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{SatelliteStepStateEvent, StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.messages.TaskStateEventHandled
import grizzled.slf4j.Logging

trait UpdateStateSupport extends BecomeWithMdc with Stash with Actor with ActorContextCreationSupport with Logging {

  import context._

  def registerStateListeners(task: Task): Unit = {
    def registerStateListener(child: ActorRef): Unit = {
      context.system.eventStream.subscribe(child, classOf[TaskStateEvent])
      context.system.eventStream.subscribe(child, classOf[StepStateEvent])
      context.system.eventStream.subscribe(child, classOf[SatelliteStepStateEvent])
    }

    val child: ActorRef = createChild(StateChangeEventListenerActor.props(task.getId, self), StateChangeEventListenerActor.name)
    registerStateListener(child)
  }

  private[tasker] def logStateChange(newState: String): Unit = debug(s">> {STATE CHANGE}: TaskManagingActor => [$newState]. Task: ${task.getId}.")

  def updateStateAndNotify(newState: TaskExecutionState, awaitHandled: Option[Receive] = None): Unit = {
    debug(s"Sending TaskStateEvent(${task.getState}->$newState) message for [taskId]")
    if (newState.isPassiveAfterExecuting && newState != DONE) {
      task.recordCompletion()
    }
    implicit val system: ActorSystem = context.system
    val unstash = (_: Unit) => {
      debug("Handled, now unstash")
      unstashAll()
    }
    awaitHandled.foreach(r => context.become(LoggingReceive {
      ReceiveOrFail("handled") {
        (r andThen unstash) orElse stashWhileAwaitHandled()
      }
    }))
    task.setTaskStateAndNotify(newState)
  }

  private[this] def stashWhileAwaitHandled(): Receive = {
    case TaskStateEventHandled(taskId, f, t) if taskId == task.getId =>
      debug(s"Received wrong state change for this task id ($f -> $t)")
    case TaskStateEventHandled(_, _, _) => // Ignore wrong handled message
    case m@_ =>
      debug(s"Stashing $m until handled seen")
      stash()
  }

  object ReceiveOrFail {
    def apply(marker: String)(receive: Receive): Receive = receive orElse {
      case m =>
        val _sender = sender()
        warn(s"Received unexpected message from ${_sender}: $m in ReceiveOrFail[$marker].")
        if (_sender != self) {
          _sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.getState}]."))
        }
    }
  }

  def createOrLookupChildForTaskBlock(): ActorRef = child("blockContainer") match {
    case Some(ref) =>
      ref
    case None =>
      createChild(PhaseContainerExecutingActor.props(task, task.getContext, self), "blockContainer")
  }

}
