package com.xebialabs.deployit.engine.tasker

import com.xebialabs.deployit.engine.tasker.log.StepLog
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, ITask}
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext
import com.xebialabs.deployit.plugin.api.services.Repository
import grizzled.slf4j.Logger
import java.io.Serializable
import java.util

class StepExecutionContext(taskExecutionContext: TaskExecutionContext,
                           step: TaskStep, task: Task, spec: TaskSpecification,
                           stepLog: StepLog)
  extends ExecutionContext with Serializable {

  val stepLogger: Logger = Logger.apply(step.getImplementation.getClass)

  if (spec != null) {
    setAttribute("config", spec.getConfig)
  }

  override def setAttribute(name: String, value: scala.AnyRef): Unit = {
    taskExecutionContext.setAttribute(name, value)
  }

  override def logOutput(output: String): Unit = {
    logOutputRaw(output + "\n")
  }

  override def logMsgOutput(key: String, args: Object*): Unit = {
    val output = getMessage(key, args:_*)
    logOutputRaw(output + "\n")
  }

  override def logOutputRaw(output: String): Unit = {
    stepLogger.info(output.stripSuffix("\n"))
    stepLog.logOutputRaw(output)
    step.touch()
  }

  override def logError(error: String): Unit = {
    logErrorRaw(error + "\n")
  }

  override def logErrorRaw(error: String): Unit = {
    logErrorRaw(error, None)
  }

  override def logError(error: String, t: Throwable): Unit = {
    logErrorRaw(error + "\n", Option(t))
  }

  def markNextAttempt(): Unit = {
    stepLog.markNextAttempt()
  }

  private def logErrorRaw(error: String, t: Option[Throwable]): Unit = {
    if (t.isDefined) {
      stepLogger.error(error.stripSuffix("\n"), t.get)
    } else {
      stepLogger.error(error.stripSuffix("\n"))
    }
    stepLog.logErrorRaw(error, t)
    step.touch()
  }

  override def getAttribute(name: String): AnyRef = taskExecutionContext.getAttributes.getOrElse(name, null) match {
    case l: OldExecutionContextListenerCleanupTrigger => l.getWrappedListener
    case l@_ => l
  }

  override def getInspectionContext: InspectionContext = spec.getInspectionContext

  override def getRepository: Repository = taskExecutionContext.getRepository

  override def getTask: ITask = new ITask {

    override def getMetadata: util.Map[String, String] = task.getMetadata

    override def getUsername: String = task.getOwner

    override def getId: String = task.getId
  }

}
