package com.xebialabs.deployit.engine.tasker

import akka.actor.{Actor, ActorLogging, ActorRef, Props, Stash}
import com.xebialabs.deployit.engine.tasker.ActiveTasksCounterActor.Messages.ForwardMessage
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.TaskStateEvent
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.collection.mutable
import scala.util.Try

object ActiveTasksCounterActor {

  object Messages {

    case class ForwardMessage(to: ActorRef, msg: Any)(implicit val sender: ActorRef)

  }

  def props = Props(new ActiveTasksCounterActor)

}

class ActiveTasksCounterActor extends Actor with Stash with ActorLogging {

  private val activeTasks = mutable.HashSet[String]()
  private val queue = mutable.Queue[ForwardMessage]()
  private val maxActiveTasks = CommonSettings(context.system).tasker.maxActiveTasks

  override def preStart(): Unit = {
    context.system.eventStream.subscribe(self, classOf[TaskStateEvent])
  }

  override def receive: Receive = collectState orElse forwardMessage

  def collectState: Receive = {
    case TaskStateEvent(taskId, _, _, state) if state.isExecutingSteps =>
      activeTasks.add(taskId)
      if (availableActiveSlots <= 0) {
        log.info("All slots for active tasks are filled. The following tasks will be QUEUED.")
        EnqueueTaskListener.stopListeners()
        context.become(collectState orElse queueMessage)
      } else {
        log.info(s"$availableActiveSlots slot(s) are available now to execute tasks.")
        EnqueueTaskListener.startListeners()
      }
    case TaskStateEvent(taskId, _, _, state) if state.isPassiveAfterExecuting =>
      activeTasks.remove(taskId)
      if (availableActiveSlots > queue.size) {
        log.info("Free slots are available to execute tasks, resuming reading task messages.")
        EnqueueTaskListener.startListeners()
        context.become(collectState orElse forwardMessage)
      }
      dequeueMessagesIfPossible()
  }

  def dequeueMessagesIfPossible(): Unit = {
    log.debug(s"$availableActiveSlots slot(s) are available now.")
    0.until(availableActiveSlots).foreach { _ =>
      Try(queue.dequeue()).toOption.foreach { case msg@ForwardMessage(to, payload) =>
        log.debug(s"dequeue $payload to ${to.path}")
        to.!(payload)(msg.sender)
      }
    }
  }

  def availableActiveSlots: Int = maxActiveTasks - activeTasks.size

  def forwardMessage: Receive = {
    case ForwardMessage(to, payload) =>
      log.debug(s"Forwarding $payload to ${to.path}")
      to.forward(payload)
  }

  def queueMessage: Receive = {
    case msg: ForwardMessage =>
      log.debug(s"queue ${msg.msg} from ${msg.sender.path}")
      queue.enqueue(msg)
  }
}
