package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Register
import com.xebialabs.deployit.engine.tasker.TaskReceiver.messages._
import com.xebialabs.deployit.engine.tasker.distribution.TaskDistributor.messages._
import com.xebialabs.deployit.engine.tasker.distribution.TasksManager
import com.xebialabs.deployit.engine.tasker.messages.{Enqueue, Registered}
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTaskRepository, PendingTaskRepository}
import grizzled.slf4j.Logging

import scala.util.Try

object TaskReceiver {
  def props(workerAddress: String, taskRepository: ActiveTaskRepository, pendingTaskRepository: PendingTaskRepository,
            workerRepository: TaskExecutionWorkerRepository, tasksManager: ActorRef): Props =
    Props(new TaskReceiver(workerAddress, taskRepository, pendingTaskRepository, workerRepository, tasksManager))

  val name = "task-receiver"

  object messages {

    case class ExecuteTask(taskId: TaskId)

  }

}

class TaskReceiver(workerAddress: String, activeTaskRepository: ActiveTaskRepository, pendingTaskRepository: PendingTaskRepository,
                   workerRepository: TaskExecutionWorkerRepository, tasksManager: ActorRef) extends Actor with Logging {
  override def receive: Receive = doReceive(Map())

  private def doReceive(specs: Map[String, TaskSpecification]): Receive = {
    case ExecuteTask(taskId) =>
      logger.info(s"Received ExecuteTask[$taskId] message.")
      extractTaskSpecification(taskId) { spec =>
        workerRepository.getWorkerByAddress(workerAddress) match {
          case Some(worker) =>
            logger.debug(s"Worker [${worker.name}] picked up the task [${taskId}]")
            tasksManager ! CreateTaskActor(spec, worker.id)
            context.become(doReceive(specs + (taskId -> spec)))
          case None =>
            logger.debug(s"Worker [$workerAddress] was not found.")
        }
      }

    case TaskActorCreated(taskActor) =>
      logger.debug(s"Received [${TaskActorCreated(taskActor)}] message.")
      taskActor ! Register()

    case Registered(taskId) =>
      logger.debug(s"Received [${Registered(taskId)}] message.")
      specs(taskId) match {
        case spec =>
          Try {
            activeTaskRepository.store(taskId, spec, workerPath(taskId))
            pendingTaskRepository.delete(taskId)
            sender() ! Enqueue(taskId)
          }.failed.foreach { ex =>
            error("Unable to move task from pending to active state.", ex)
          }
          context.become(doReceive(specs - taskId))
        case _ =>
      }
  }

  private def workerPath(taskId: TaskId) = ActorPath.fromString(s"$workerAddress/user/${TasksManager.name}/$taskId")

  private def extractTaskSpecification(taskId: TaskId)(func: TaskSpecification => Unit): Unit = {
    pendingTaskRepository.task(taskId, loadFullSpec = true) match {
      case Some(pendingTask) =>
        pendingTask.spec match {
          case Some(taskSpecification) => func(taskSpecification)
          case None => onTaskNotFound(taskId)
        }
      case None => onTaskNotFound(taskId)
    }

    def onTaskNotFound(taskId: TaskId): Unit = {
      EnqueueTaskListener.startListeners()
      warn(s"Pending task [$taskId] cannot be found. Was it force-canceled or manually deleted from the database?")
    }
  }
}
