package com.xebialabs.deployit.engine.tasker

import akka.actor.{Actor, ActorIdentity, ActorRef, ActorSelection, DeadLetter, Identify, Props, Terminated}
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState.FAILED
import com.xebialabs.deployit.engine.tasker.ArchivedListeningActor.{Forward, ForwardToRef}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.TaskStateEvent
import com.xebialabs.deployit.engine.tasker.messages.{Archived, Cancelled, FailedToArchive}

import scala.concurrent.Promise

class ArchivedListeningActor(taskId: TaskId, promise: Promise[TaskId]) extends Actor {

  import context._

  def receive: Actor.Receive = {
    case Forward(actor, message) =>
      become(identifyActor(message))
      actor ! Identify("")
    case ForwardToRef(actor, message) =>
      become(identifyActor(message))
      actor ! Identify("")
  }

  def identifyActor(originalMessage: AnyRef): Actor.Receive = {
    case ActorIdentity(_, Some(actorRef)) =>
      watch(actorRef)
      system.eventStream.subscribe(self, classOf[DeadLetter])
      become(await(actorRef))
      system.eventStream.subscribe(self, classOf[TaskStateEvent])
      actorRef ! originalMessage
    case _ =>
      promise.failure(new TaskNotFoundException("akka system", taskId))
  }

  def await(actorRef: ActorRef): Actor.Receive = {
    case Terminated(`actorRef`) | DeadLetter(_, `self`, `actorRef`) =>
      promise.tryFailure(new TaskerException(s"Task $taskId was terminated by a different process"))
    case FailedToArchive(`taskId`, exception) =>
      forget(actorRef)
      promise.tryFailure(new TaskerException(exception, s"Task [$taskId] failed to archive"))
    case akka.actor.Status.Failure(exception) =>
      forget(actorRef)
      promise.tryFailure(new TaskerException(exception, s"Task [$taskId] failed to archive"))
    case Archived(`taskId`) | Cancelled(`taskId`) =>
      forget(actorRef)
      promise.trySuccess(taskId)
    case TaskStateEvent(`taskId`, _, prev, FAILED) =>
      forget(actorRef)
      promise.tryFailure(new TaskerException(s"Task [$taskId] went from $prev to $FAILED while cancelling"))
  }

  private def forget(actorRef: ActorRef): Unit = {
    unwatch(actorRef)
    system.eventStream.unsubscribe(self)
  }
}

object ArchivedListeningActor {
  def props(taskId: TaskId, promise: Promise[TaskId]) = Props(new ArchivedListeningActor(taskId, promise))

  case class ForwardToRef(actorRef: ActorRef, message: AnyRef)
  case class Forward(actor: ActorSelection, message: AnyRef)

}
