package com.xebialabs.deployit.tasksystem

import com.typesafe.config.ConfigFactory._
import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.security.SecretKeyHolder
import com.xebialabs.xlplatform.config._
import com.xebialabs.xlplatform.settings.{CommonSettings, ConfigSupport, SecuritySettings}

import scala.jdk.CollectionConverters._

object ConfigLoading extends ConfigSupport {

  def loadConfig(systemConfig: Config): Config = {

    val commonSettings = new CommonSettings(systemConfig.withFallback(ConfigFactory.defaultReference()))

    val satelliteEnabled = commonSettings.satellite.enabled
    val remotingEnabled = satelliteEnabled || !commonSettings.inProcessTaskEngine
    val remotingDisabled = !remotingEnabled
    val serializationType = commonSettings.serialization.serializationType

    val conditionalFallbacks: Seq[(Config, Boolean)] = Seq(
      hostAddressConfiguration("server.hostname") -> remotingEnabled,
      parseMap(Map("server.port" -> ServerConfiguration.getInstance().getServerPort().toString).asJava) -> remotingEnabled,
      sslRemoteTransports(commonSettings.security) -> remotingEnabled,
      parseResources("remote-enabled.conf") -> remotingEnabled,
      parseResources(s"$serializationType-pekko-serializers.conf") -> true,
      parseResources("pekko-dispatchers.conf") -> true,
      ConfigLoader.loadWithDynamic("xl-deploy.conf") -> remotingDisabled,
      parseResources("satellite-enabled.conf") -> satelliteEnabled,
    )
    loadWithFallbacks(
      systemConfig,
      conditionalFallbacks
        .filter(_._2)
        .map(_._1)  : _*
    ).withFallback(ConfigFactory.defaultReference())
      .decrypted(SecretKeyHolder.get())
  }

  private[this] def sslRemoteTransports(security: SecuritySettings): Config =
    parseMap(
      Map("pekko.remote.artery.transport" -> (if (security.enabled) "tls-tcp" else "tcp"))
        .asJava
    )
}

