package com.xebialabs.xlplatform.test.tasksystem

import java.io.IOException
import java.net.ServerSocket
import akka.actor.ActorSystem
import akka.testkit.TestKit
import com.typesafe.config.ConfigFactory
import com.xebialabs.deployit.tasksystem.TaskActorSystem
import com.xebialabs.xlplatform.config.{ConfigurationHolder, DefaultProductConfiguration}
import org.scalatest.{BeforeAndAfterEach, Suite}

import java.util.UUID
import scala.concurrent.duration._
import scala.util.Random
import scala.jdk.CollectionConverters._

trait Portfinder {

  def findFreePort(): Int = {
    val socket: ServerSocket = null
    try {
      val socket = new ServerSocket(0)
      socket.setReuseAddress(true)
      val port = socket.getLocalPort
      try {
        socket.close()
      } catch  {
        case _: IOException => // Ignore IOException on close()
      }
      return port
    } catch {
      case _: IOException =>
    } finally {
      if (socket != null) {
        try {
          socket.close()
        } catch {
          case _: IOException =>
        }
      }
    }
    throw new IllegalStateException("Could not find a free TCP/IP port to start Integration Test Server")
  }

}

trait RemoteTestActorSystem extends BeforeAndAfterEach with Portfinder with SecuritySupport {
  this: Suite =>

  //For some of the test that result in calling SatelliteCommunicatorSystem.newActorSystem
  // without overriding default config it's needed to inject random ports via system properties
  System.setProperty("test.port", findFreePort().toString)
  System.setProperty("test.ssl.port", findFreePort().toString)

  initPasswordEncrypter()
  new ConfigurationHolder(new DefaultProductConfiguration).init()
  var satelliteClientPort: Int = 0

  implicit var actorSystem: ActorSystem = _

  override protected def beforeEach(): Unit = {
    satelliteClientPort = findFreePort()
    new ConfigurationHolder(new DefaultProductConfiguration).init()
    actorSystem = TaskActorSystem.newActorSystem(ConfigFactory.parseMap(
      Map(
        "deploy.satellite.enabled" -> "yes",
        "deploy.satellite.timeout.ping" -> "1 second",
        "deploy.satellite.timeout.remoteAsk" -> "1 second",
        "deploy.server.port" -> satelliteClientPort.toString,
        "deploy.server.algorithms.checksum" -> "SHA-256"
      ).asJava),
      Some(s"${this.getClass.getSimpleName}-task-system-${UUID.randomUUID()}")
    )

    super.beforeEach()
  }

  override protected def afterEach(): Unit = {
    super.afterEach()
    TestKit.shutdownActorSystem(actorSystem, 60.seconds, true)
  }
}

trait RemoteTestActorSystemConstructor extends Portfinder {
  def defaultConfig: Map[String,Any] = Map(
    "deploy.satellite.enabled" -> "yes",
    "deploy.satellite.timeout.remoteAsk" -> "1 second",
    "deploy.satellite.timeout.ping" -> "1 second",
    "deploy.server.port" -> findFreePort()
  )

  def newActorSystemWithConf(customConfig: Map[String, Any]): ActorSystem = {
    val mergedConfig = defaultConfig ++ customConfig
    TaskActorSystem.newActorSystem(
      ConfigFactory.parseMap(mergedConfig.asJava),
      Option(s"${this.getClass.getSimpleName}-task-system-${Random.nextInt(1000)}")
    )
  }
}
