package ai.digital.deploy.task.serdes

import ai.digital.deploy.task.serdes.DeployJacksonObjectMapperFactory.{getInstance, loadAndGetInstance}
import ai.digital.deploy.task.serdes.deploy.DeployModule
import ai.digital.deploy.task.serdes.overthere.OverthereModule
import ai.digital.deploy.task.serdes.planner.PlannerModule
import ai.digital.deploy.task.serdes.tasker.TaskerModule
import akka.serialization.jackson.JacksonObjectMapperFactory
import com.fasterxml.jackson.annotation.{JsonAutoDetect, PropertyAccessor}
import com.fasterxml.jackson.core.JsonFactory
import com.fasterxml.jackson.databind
import com.fasterxml.jackson.databind.{MapperFeature, Module, ObjectMapper}
import org.springframework.security.core.{Authentication, GrantedAuthority}
import org.springframework.util.ClassUtils

class DeployJacksonObjectMapperFactory extends JacksonObjectMapperFactory {

  private val springSecurityModuleClassnames = Array(
    "org.springframework.security.jackson2.CoreJackson2Module",
    "org.springframework.security.cas.jackson2.CasJackson2Module",
    "org.springframework.security.web.jackson2.WebJackson2Module",
    "org.springframework.security.web.server.jackson2.WebServerJackson2Module",
    "org.springframework.security.web.jackson2.WebServletJackson2Module",
    "org.springframework.security.oauth2.client.jackson2.OAuth2ClientJackson2Module",
  )

  private val deployModuleClasses = Array(
    classOf[DeployModule],
    classOf[OverthereModule],
    classOf[PlannerModule],
    classOf[TaskerModule],
  )

  override def newObjectMapper(bindingName: String, jsonFactory: JsonFactory): ObjectMapper =
    super.newObjectMapper(bindingName, jsonFactory)
      .addMixIn(classOf[grizzled.slf4j.Logger], classOf[MixinForIgnoreType])
      .addMixIn(classOf[org.slf4j.Logger], classOf[MixinForIgnoreType])
      .addMixIn(classOf[Authentication], classOf[MixinWithJsonTypeInfo])
      .addMixIn(classOf[GrantedAuthority], classOf[MixinWithJsonTypeInfo])
      .addHandler(DeployDeserializationProblemHandler)

  override def overrideConfiguredModules(bindingName: String, configuredModules: Seq[databind.Module]): Seq[databind.Module] = {
    val springSecurityModules = springSecurityModuleClassnames.flatMap(loadAndGetInstance)
    val deployModules = deployModuleClasses.map(getInstance(_))
    super.overrideConfiguredModules(bindingName,
      configuredModules
        ++ springSecurityModules
        ++ deployModules
    )
  }

  override def overrideConfiguredMapperFeatures(bindingName: String, configuredFeatures: Seq[(MapperFeature, Boolean)]): Seq[(MapperFeature, Boolean)] =
    super.overrideConfiguredMapperFeatures(bindingName,
      configuredFeatures ++ Seq(
        MapperFeature.PROPAGATE_TRANSIENT_MARKER -> true,
      )
    )

  override def overrideConfiguredVisibility(bindingName: String, configuredFeatures: Seq[(PropertyAccessor, JsonAutoDetect.Visibility)]): Seq[(PropertyAccessor, JsonAutoDetect.Visibility)] =
    super.overrideConfiguredVisibility(bindingName,
      configuredFeatures ++ Seq(
        PropertyAccessor.GETTER -> JsonAutoDetect.Visibility.NONE,
        PropertyAccessor.IS_GETTER -> JsonAutoDetect.Visibility.NONE,
        PropertyAccessor.SETTER -> JsonAutoDetect.Visibility.NONE,
      )
    )
}

object DeployJacksonObjectMapperFactory {

  private def getInstance(clazz: Class[_]): Module =
    clazz.asInstanceOf[Class[_ <: Module]].newInstance()

  def loadClass(className: String): Option[Class[_]] =
    try {
      Some(ClassUtils.forName(className, this.getClass.getClassLoader))
    } catch {
      case _: ClassNotFoundException => None
    }

  private def loadAndGetInstance(className: String): Option[Module] =
    try {
      val securityModule = loadClass(className)
      securityModule.map(getInstance _)
    } catch {
      case _: ClassNotFoundException => None
    }
}
