package ai.digital.deploy.task.serdes.yaml

import com.fasterxml.jackson.annotation.{JsonCreator, JsonIdentityInfo, JsonIgnore, JsonProperty, JsonTypeInfo, ObjectIdGenerator}
import com.fasterxml.jackson.core.{JsonGenerator, JsonParser}
import com.fasterxml.jackson.databind.{DeserializationContext, JsonDeserializer, JsonNode, JsonSerializer, KeyDeserializer, ObjectMapper, SerializerProvider}
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.xebialabs.deployit.booter.local.LocalDescriptorRegistryId
import com.xebialabs.deployit.booter.remote.BooterConfig
import com.xebialabs.deployit.engine.tasker.{BlockPath, PhaseContainer}
import com.xebialabs.deployit.io.WorkDirInContext
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistryId
import com.xebialabs.deployit.plugin.api.udm.artifact.{DerivedArtifact, SourceArtifact}
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, OnTaskFailurePolicy, OnTaskSuccessPolicy}
import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.xltype.serialization.xstream.XStreamReaderWriter
import org.apache.http.HttpRequestInterceptor
import org.springframework.beans.factory.annotation.{Autowired, Configurable, Qualifier}
import org.springframework.security.core.Authentication
import java.io.File
import java.util.concurrent.atomic.AtomicInteger

package object tasker {

  class CiIdGenerator extends ObjectIdGenerator[String] {

    override def getScope: Class[_] = classOf[Object]

    override def canUseFor(gen: ObjectIdGenerator[_]): Boolean =
      (gen.getClass == getClass) && (gen.getScope == getScope)

    override def forScope(scope: Class[_]): ObjectIdGenerator[String] = this

    override def newForSerialization(context: Any): ObjectIdGenerator[String] = this

    override def key(key: Any): ObjectIdGenerator.IdKey =
      if (key == null)
        null
      else
        new ObjectIdGenerator.IdKey(getClass, getScope, key)

    override def generateId(forPojo: Any): String =
      forPojo match {
        case ci: ConfigurationItem => ci.getId + "." + ci.getType.getName
        case _ => throw new IllegalArgumentException("Not supported " + forPojo);
      }
  }

  @JsonIdentityInfo(
    generator = classOf[CiIdGenerator],
    property = "__id"
  )
  abstract class MixinForConfigurationItem extends MixinWithJsonTypeInfo {}

  @JsonCreator
  abstract class MixinForTaskSpecification(@JsonProperty("id") id: String,
                                           @JsonProperty("description") description: String,
                                           @JsonProperty("owner") owner: Authentication,
                                           @JsonProperty("workDir") workDir: WorkDir,
                                           @JsonProperty("block") block: PhaseContainer,
                                           @JsonProperty("onSuccessPolicy") onSuccessPolicy: OnTaskSuccessPolicy,
                                           @JsonProperty("onFailurePolicy") onFailurePolicy: OnTaskFailurePolicy) {}

  @JsonCreator
  abstract class MixinForTaskStep(@JsonProperty("implementation") implementation: Step,
                                  @JsonProperty("skippable") skippable: Boolean,
                                  @JsonProperty("failureCount") failureCount: AtomicInteger,
                                  @JsonProperty("description") description: String) {}

  @JsonCreator
  abstract class MixinForWorkdirCleanerTrigger(@JsonProperty("workDir") workDir: WorkDir) {}

  @JsonCreator
  abstract class MixinForDerivedArtifactFile(@JsonProperty("derivedArtifact") derivedArtifact: DerivedArtifact[_ <: SourceArtifact]) {}

  @Configurable // or use SpringHandlerInstantiator in Jackson ObjectMapper
  class WorkDirDeserializer extends JsonDeserializer[WorkDir] {

    @Autowired
    @Qualifier("baseWorkDir")
    val baseWorkDir: File = new File("work")

    override def deserialize(jp: JsonParser, ctxt: DeserializationContext): WorkDir = {
      val mapper = jp.getCodec.asInstanceOf[ObjectMapper]
      val jsonNode: JsonNode = mapper.readTree(jp)
      val workDirNode = readJsonNode(jsonNode, "workDir")
      val workDir = if (workDirNode.isNull) {
        val prefix = readJsonNode(jsonNode, "prefix").asText()
        new WorkDir(baseWorkDir, prefix)
      } else {
        val remoteBaseWorkDir = mapper.readValue(workDirNode.traverse(mapper), classOf[LocalFile])
        new WorkDir(remoteBaseWorkDir)
      }
      workDir.rebase(baseWorkDir, true)
    }
  }

  @JsonDeserialize(using = classOf[WorkDirDeserializer])
  abstract class MixinForWorkDir {
    @JsonIgnore
    private var baseWorkDir: LocalFile = _
  }

  class DescriptorRegistryIdDeserializer extends JsonDeserializer[DescriptorRegistryId] {

    private lazy val localDescriptorRegistryId = new LocalDescriptorRegistryId()

    override def deserialize(jp: JsonParser, ctxt: DeserializationContext): DescriptorRegistryId = {
      val mapper = jp.getCodec.asInstanceOf[ObjectMapper]
      val jsonNode: JsonNode = mapper.readTree(jp)
      val idNode = readJsonNode(jsonNode, "id")
      if (!idNode.isNull && "LOCAL" == idNode.asText()) {
        localDescriptorRegistryId
      } else {
        val builder = new BooterConfig.Builder
        builder
          .withProtocol(BooterConfig.Protocol.valueOf(readJsonNode(jsonNode, "protocol").asText(BooterConfig.Protocol.HTTP.name())))
          .withHost(readJsonNode(jsonNode, "host").asText())
          .withPort(readJsonNode(jsonNode, "port").asInt(-1))
          .withProxyHost(readJsonNode(jsonNode, "proxyHost").asText())
          .withProxyPort(readJsonNode(jsonNode, "proxyPort").asInt(-1))
          .withContext(readJsonNode(jsonNode, "context").asText("deployit"))
          .withCredentials(readJsonNode(jsonNode, "username").asText(), readJsonNode(jsonNode, "password").asText())
        builder.build()
      }
    }
  }

  @JsonDeserialize(using = classOf[DescriptorRegistryIdDeserializer])
  abstract class MixinForDescriptorRegistryId {}

  abstract class MixinForBooterConfig {
    @JsonIgnore
    private var contextRoot: String = _
    @JsonIgnore
    private var connectionPoolSize: Int = _
    @JsonIgnore
    private var socketTimeout: Int = _
    @JsonIgnore
    private var httpRequestInterceptors: List[HttpRequestInterceptor] = _
    @JsonIgnore
    private var xStreamReaderWriter: XStreamReaderWriter = _
  }

  @JsonTypeInfo(
    use = JsonTypeInfo.Id.MINIMAL_CLASS,
    include = JsonTypeInfo.As.PROPERTY,
    defaultImpl = classOf[WorkDirInContext],
    property = "__dt")
  abstract class MixinForSupplierWorkDir {}

  class BlockPathSerializer extends JsonSerializer[BlockPath] {
    override def serialize(value: BlockPath, gen: JsonGenerator, serializers: SerializerProvider): Unit = {
      gen.writeString(value.toBlockId)
    }
  }

  class BlockPathKeyDeserializer extends KeyDeserializer {
    override def deserializeKey(key: String, ctxt: DeserializationContext): AnyRef = {
      BlockPath(key)
    }
  }

  class BlockPathDeserializer extends JsonDeserializer[BlockPath] {
    override def deserialize(p: JsonParser, ctxt: DeserializationContext): BlockPath = {
      BlockPath(p.getText)
    }
  }
}
