package io.altoo.serialization.kryo.pekko

import ai.digital.deploy.task.serdes.TaskPekkoSerializer
import ai.digital.deploy.task.serdes.kryo.{KryoSerializer, WorkDirSerializer}
import com.esotericsoftware.kryo.kryo5.Kryo
import com.xebialabs.deployit.repository.WorkDir
import io.altoo.serialization.kryo.scala.{DefaultQueueBuilder, KryoSerializationSettings, KryoSerializerBackend, SerializerPool}
import org.apache.pekko.actor.ExtendedActorSystem
import org.slf4j.LoggerFactory

class KryoPekkoSerializer(system: ExtendedActorSystem) extends PekkoKryoSerializer(system) with TaskPekkoSerializer {

  private val serializerPoolOnSuper = this.getClass.getSuperclass.getSuperclass.getDeclaredField("serializerPool")
  serializerPoolOnSuper.setAccessible(true)

  private val queueBuilderClassOnSuper = this.getClass.getSuperclass.getSuperclass.getDeclaredField("queueBuilderClass")
  queueBuilderClassOnSuper.setAccessible(true)
  private val queueBuilderClass = queueBuilderClassOnSuper.get(this).asInstanceOf[Class[_]]

  private val settingsOnSuper = this.getClass.getSuperclass.getSuperclass.getDeclaredField("settings")
  settingsOnSuper.setAccessible(true)
  private val settings = settingsOnSuper.get(this).asInstanceOf[KryoSerializationSettings]

  private val pool = new SerializerPool(
    queueBuilderClass.getDeclaredConstructor().newInstance().asInstanceOf[DefaultQueueBuilder],
    () => new KryoSerializerBackend(
      getCustomKryo,
      settings.bufferSize,
      settings.maxBufferSize,
      this.useManifest,
      settings.useUnsafe
    )(LoggerFactory.getLogger(getClass), system.dynamicAccess.classLoader)
  )

  serializerPoolOnSuper.set(this, pool)

  var kryoCustomizers: Seq[KryoCustomizer] = Seq()

  def setKryoCustomizers(kryoCustomizer: KryoCustomizer*): Unit = {
    kryoCustomizers = kryoCustomizer
  }

  private def getCustomKryo: Kryo = {
    val kryo = KryoSerializer.getMagicCombination(system)
    kryoCustomizers.foreach(_.customize(kryo))
    kryo
  }

  override def initExternalWorker(conf: TaskPekkoSerializer.Conf, createWorkDir: Boolean = true): Unit = {
    setKryoCustomizers(
      (kryo: Kryo) => kryo.addDefaultSerializer(classOf[WorkDir], new WorkDirSerializer(kryo, conf.baseWorkDir, createWorkDir))
    )
  }
}

trait KryoCustomizer {
  def customize(kryo: Kryo): Unit
}
