package ai.digital.deploy.task.serdes.kryo


import com.esotericsoftware.kryo.Kryo
import com.esotericsoftware.kryo.io.{Input, Output}
import org.agrona.concurrent.ManyToManyConcurrentArrayQueue
import org.apache.pekko.serialization.{Serializer => PekkoSerializer}

import java.io.{ByteArrayInputStream, ByteArrayOutputStream}
import scala.reflect.ClassTag

trait BaseMessageSerializer[T <: AnyRef] extends PekkoSerializer {

  private class KryoPool(newInstance: () => Kryo) {

    private val pool = new ManyToManyConcurrentArrayQueue[Kryo](Runtime.getRuntime.availableProcessors * 4)

    def fetch(): Kryo =
      pool.poll() match {
        case o if o != null => o
        case null => newInstance()
      }

    def release(o: Kryo): Unit = {
      o.reset()
      pool.offer(o)
    }
  }

  def kryoFactory(): Kryo

  val classTag: ClassTag[T]

  private val serializerPool = new KryoPool(kryoFactory)

  override def fromBinary(bytes: Array[Byte], manifest: Option[Class[_]]): AnyRef = {
    val input = new Input(new ByteArrayInputStream(bytes))

    val kryo = serializerPool.fetch()
    try {
      kryo.readObject(input, classTag.runtimeClass.asInstanceOf[Class[AnyRef]])
    } finally {
      serializerPool.release(kryo)
    }
  }

  override def toBinary(obj: AnyRef): Array[Byte] = {
    import com.xebialabs.xlplatform.utils.ResourceManagement._
    val stream = new ByteArrayOutputStream()
    using(new Output(stream)) { output =>
      val kryo = serializerPool.fetch()
      try {
        kryo.writeObject(output, obj)
        output.flush()
      } finally {
        serializerPool.release(kryo)
      }
    }
    stream.toByteArray
  }
}
