package com.xebialabs.deployit.plugin.satellite

import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol._
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings
import org.apache.pekko.actor.{ActorSelection, ActorSystem}
import org.apache.pekko.pattern.ask
import org.apache.pekko.util.Timeout

import scala.beans.BeanProperty
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContextExecutor, Future, Promise}
import scala.util.{Success, Try}

abstract class SatelliteStep(system: ActorSystem) extends SatelliteActorSystemStep(system) with AwaitForever {
  @transient implicit lazy val dispatcher: ExecutionContextExecutor = satelliteCommunicatorSystem.dispatcher

  var stepActorLocator: ActorLocator = _
  var stepSatelliteName: String = _

  import ForceRestartSatelliteStep.ForceRetryOnceMore

  def executeTry(actorLocator: ActorLocator,
                 satelliteName: String,
                 satelliteCommunicatorSystem: ActorSystem,
                 force: Boolean = false)
                (implicit ctx: ExecutionContext): Try[StepExitCode] = Try {
    this.stepSatelliteName = satelliteName
    this.stepActorLocator = actorLocator

    blockOn(satelliteStatus).get match {
      case SatelliteRunning(since) =>
        restartSatellite(Option(since), force)
      case Restarting =>
        Future.successful(StepExitCode.SUCCESS)
      case WaitingRestart(_) =>
        restartSatellite(None, force)
    }

    Await.result(restartSatellite(force = force), atMost = Duration.Inf)
  }

  def locateRemoteActor: ActorSelection = {
    stepActorLocator.locate(Paths.satelliteControl)(satelliteCommunicatorSystem)
  }

  private def satelliteStatus: Future[SatelliteStatus] = {
    implicit val satelliteTimeout: Timeout = CommonSettings(satelliteCommunicatorSystem).satellite.remoteAskTimeout
    (locateRemoteActor ? GetSatelliteStatus).mapTo[SatelliteStatus]
  }

  private def restartSatellite(previousSatelliteRunningSince: Option[Long] = None, force: Boolean)(implicit ctx: ExecutionContext): Future[StepExitCode] = {

    implicit val satelliteTimeout: Timeout = CommonSettings(satelliteCommunicatorSystem).satellite.remoteAskTimeout
    val result = Promise[StepExitCode]()

    def satelliteRestarting(): Unit = {
      result.success(StepExitCode.SUCCESS)
      ctx.logOutput(s"Satellite ${this.stepSatelliteName} is restarting")
    }

    def satelliteStillActive(runningTasks: Seq[TaskId]): Unit = {
      ctx.logOutput(s"Satellite ${this.stepSatelliteName} is running ${runningTasks.size} tasks: ${runningTasks.mkString}")
      result.failure(ForceRetryOnceMore)
    }

    def sendRestartMessage(force: Boolean): Unit = {
      ctx.logOutput(s"Resending ${if (force) "force" else "soft"} restart message to satellite ${this.stepSatelliteName}")
      locateRemoteActor ! (if (force) ForceRestartSatellite else RestartSatellite)
      result.failure(ForceRetryOnceMore)
    }

    satelliteCommunicatorSystem.scheduler.scheduleOnce(1.second) {
      satelliteStatus.onComplete {
        case Success(SatelliteRunning(since)) if Option(since) != previousSatelliteRunningSince =>
          satelliteRestarting()
        case Success(SatelliteRunning(since)) if Option(since) == previousSatelliteRunningSince =>
          sendRestartMessage(force)
        case Success(WaitingRestart(runningTasks)) =>
          if (force) {
            sendRestartMessage(force)
          } else {
            satelliteStillActive(runningTasks)
          }
        case Success(Restarting) =>
          satelliteRestarting()
        case _ =>
          ctx.logOutput(s"Could not contact satellite ${this.stepSatelliteName}, retrying...")
          result.failure(ForceRetryOnceMore)
      }
    }

    result.future.recoverWith {
      case ForceRetryOnceMore => restartSatellite(previousSatelliteRunningSince, force)
    }
  }
}

object ConditionalRestartSatelliteStep {

  private object RetryOnceMore extends Throwable

  def apply(satelliteCi: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem): ConditionalRestartSatelliteStep = {
    new ConditionalRestartSatelliteStep(
      satelliteName = satelliteCi.getName,
      actorLocator = ActorLocator(satelliteCi),
      description = s"Conditional restart satellite ${satelliteCi.getName}",
      satelliteCommunicatorSystem)
  }
}

object ForceRestartSatelliteStep {

  object ForceRetryOnceMore extends Throwable

  def apply(satelliteCi: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem): ForceRestartSatelliteStep = {
    new ForceRestartSatelliteStep(
      satelliteName = satelliteCi.getName,
      actorLocator = ActorLocator(satelliteCi),
      description = s"Force restart satellite ${satelliteCi.getName}",
      satelliteCommunicatorSystem)
  }
}

object SoftRestartSatelliteStep {

  object SoftRetryOnceMore extends Throwable

  def apply(satelliteCi: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem): SoftRestartSatelliteStep = {
    new SoftRestartSatelliteStep(
      satelliteName = satelliteCi.getName,
      actorLocator = ActorLocator(satelliteCi),
      description = s"Soft restart satellite ${satelliteCi.getName}",
      satelliteCommunicatorSystem)
  }
}

case class ForceRestartSatelliteStep(var satelliteName: String, var actorLocator: ActorLocator, @BeanProperty description: String, @transient system: ActorSystem) extends SatelliteStep(system) {

  @BeanProperty val order: Int = Step.DEFAULT_ORDER

  override def execute(ctx: ExecutionContext): StepExitCode = {
    executeTry(actorLocator, satelliteName, satelliteCommunicatorSystem, force = true)(ctx).recover {
      case ex =>
        ctx.logError(s"Error: ${ex.getMessage}", ex)
        StepExitCode.FAIL
    }.get
  }
}

case class SoftRestartSatelliteStep(var satelliteName: String, var actorLocator: ActorLocator, @BeanProperty description: String, @transient system: ActorSystem) extends SatelliteStep(system) {

  @BeanProperty val order: Int = Step.DEFAULT_ORDER

  override def execute(ctx: ExecutionContext): StepExitCode = {
    executeTry(actorLocator, satelliteName, satelliteCommunicatorSystem)(ctx).recover {
      case ex =>
        ctx.logError(s"Error: ${ex.getMessage}", ex)
        StepExitCode.FAIL
    }.get
  }
}

case class ConditionalRestartSatelliteStep(var satelliteName: String, var actorLocator: ActorLocator, @BeanProperty description: String, @transient system: ActorSystem) extends SatelliteStep(system) {

  @BeanProperty val order: Int = Step.DEFAULT_ORDER

  override def execute(ctx: ExecutionContext): StepExitCode = {
    Option(ctx.getAttribute("restartRequired")) match {
      case Some(_) => executeTry(actorLocator, satelliteName, satelliteCommunicatorSystem)(ctx).recover {
        case ex =>
          ctx.logError(s"Error: ${ex.getMessage}", ex)
          StepExitCode.FAIL
      }.get

      case None =>
        ctx.logOutput(s"Satellite $satelliteName is up to date, so it does not need to be restarted.")
        StepExitCode.SUCCESS
    }
  }
}
